package com.devexperts.mdd.auth.entitle;

import com.devexperts.auth.AuthToken;
import com.devexperts.logging.Logging;
import com.devexperts.mdd.auth.util.SignedToken;
import com.devexperts.qd.qtp.MessageAdapterConnectionFactory;
import com.devexperts.qd.qtp.auth.QDLoginHandler;
import com.devexperts.qd.qtp.auth.QDLoginHandlerFactory;
import com.devexperts.util.InvalidFormatException;
import com.dxfeed.promise.Promise;

import java.time.Duration;

/**
 * LoginHandler factory that allows to send signed token to the remote side for QD communication.
 *
 * <p>Connection string must be specified as: {@code <address>[auth=entitle:<token>]}, where {@code <token>}
 * is a signed token given to the user after successful login. Alternatively address can be specified as
 * {@code <address>[auth=entitle]} and use {@link #setAppToken(String)} method to specify application token.
 */
public class EntitleLoginHandlerFactory implements QDLoginHandlerFactory {

    /** Name of the scheme to be used for authenticating clients. */
    public static final String ENTITLE_SCHEME = "entitle";

    private static final Logging log = Logging.getLogging(EntitleLoginHandlerFactory.class);

    public static volatile String appToken;

    public EntitleLoginHandlerFactory() {
        log.info("Registering auth scheme '" + ENTITLE_SCHEME + "'");
    }

    /**
     * Returns global application token string.
     * @return token string.
     */
    public static String getAppToken() {
        return appToken;
    }

    /**
     * Sets global application token string.
     * @param appToken
     */
    public static void setAppToken(String appToken) {
        EntitleLoginHandlerFactory.appToken = appToken;
    }

    /**
     * Create and return token for the specified session and user, signed by the specified secret
     *
     * @param issuer company owning sessions (sent as an issuer in the token)
     * @param sessionType type of session (sent as a subject in the token)
     * @param user user GUID (sent in the message field)
     * @param secret random string used to sign the token (must be kept secret from client code)
     * @return signed token
     */
    public static String createToken(String issuer, String sessionType, String user, String secret) {
        return SignedToken.newBuilder()
            .setIssuer(issuer)
            .setSubject(sessionType)
            .setMessage(user)
            .setIssuedNow()
            .setExpirationFromNow(Duration.ofDays(1))
            .toToken()
            .signToken(secret);
    }

    @Override
    public QDLoginHandler createLoginHandler(String login, MessageAdapterConnectionFactory factory)
        throws InvalidFormatException
    {
        if (!login.startsWith(ENTITLE_SCHEME)) {
            return null;
        }

        boolean hasToken = login.length() > ENTITLE_SCHEME.length();
        if (hasToken && login.charAt(ENTITLE_SCHEME.length()) != ':') {
            log.error("Auth scheme '" + ENTITLE_SCHEME + "' login must be specified as: "
                + ENTITLE_SCHEME + "[:<token>]");
            return null;
        }

        String token = hasToken ? login.substring(ENTITLE_SCHEME.length() + 1) : getAppToken();
        if (token == null || token.isEmpty()) {
            log.error("No token available for scheme '" + ENTITLE_SCHEME + "'");
            return null;
        }

        log.info("Using auth scheme '" + ENTITLE_SCHEME + "' with token: " + token);
        return new AutherLoginHandler(token);

    }

    protected static class AutherLoginHandler implements QDLoginHandler {
        private final AuthToken token;

        public AutherLoginHandler(String token) {
            this.token = AuthToken.createBearerToken(token);
        }

        @Override
        public Promise<AuthToken> login(String reason) {
            if (reason != null) {
                log.error("Server rejected token: " + reason);
                return Promise.failed(new SecurityException(reason));
            } else {
                return Promise.completed(token);
            }
        }

        @Override
        public AuthToken getAuthToken() {
            return token;
        }
    }
}
