/*
 * !++
 * QDS - Quick Data Signalling Library
 * !-
 * Copyright (C) 2002 - 2024 Devexperts LLC
 * !-
 * This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
 * If a copy of the MPL was not distributed with this file, You can obtain one at
 * http://mozilla.org/MPL/2.0/.
 * !__
 */
package com.dxfeed.api.experimental.model;

import com.devexperts.qd.util.DxTimer;
import com.dxfeed.api.DXFeed;
import com.dxfeed.api.impl.DXFeedImpl;
import com.dxfeed.event.IndexedEvent;
import com.dxfeed.event.IndexedEventSource;
import com.dxfeed.event.market.Order;
import com.dxfeed.event.market.OrderBase;
import com.dxfeed.event.market.OrderSource;
import com.dxfeed.event.market.Scope;
import com.dxfeed.event.market.Side;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class MarketDepthModel<E extends OrderBase> implements AutoCloseable {
    private static final Comparator<? super OrderBase> ORDER_COMPARATOR = (o1, o2) -> {
        boolean ind1 = o1.getScope() == Scope.ORDER;
        boolean ind2 = o2.getScope() == Scope.ORDER;
        if (ind1 && ind2) {
            // Both orders are individual orders
            int c = Long.compare(o1.getTimeSequence(), o2.getTimeSequence()); // asc
            if (c != 0)
                return c;
            c = Long.compare(o1.getIndex(), o2.getIndex()); // asc
            return c;
        } else if (ind1) {
            // First order is individual, second is not
            return 1;
        } else if (ind2) {
            // Second order is individual, first is not
            return -1;
        } else {
            // Both orders are non-individual orders
            int c = Double.compare(o2.getSizeAsDouble(), o1.getSizeAsDouble()); // desc
            if (c != 0)
                return c;
            c = Long.compare(o1.getTimeSequence(), o2.getTimeSequence()); // asc
            if (c != 0)
                return c;
            c = o1.getScope().getCode() - o2.getScope().getCode(); // asc
            if (c != 0)
                return c;
            c = o1.getExchangeCode() - o2.getExchangeCode(); // asc
            if (c != 0)
                return c;
            if (o1 instanceof Order && o2 instanceof Order) {
                Order order1 = (Order) o1;
                Order order2 = (Order) o2;
                c = compareString(order1.getMarketMaker(), order2.getMarketMaker()); // asc
                if (c != 0)
                    return c;
            }
            c = Long.compare(o1.getIndex(), o2.getIndex()); // asc
            return c;
        }
    };

    private static final Comparator<? super OrderBase> BUY_COMPARATOR = (o1, o2) ->
        o1.getPrice() < o2.getPrice() ? 1 : // desc
            o1.getPrice() > o2.getPrice() ? -1 :
                ORDER_COMPARATOR.compare(o1, o2);

    private static final Comparator<? super OrderBase> SELL_COMPARATOR = (o1, o2) ->
        o1.getPrice() < o2.getPrice() ? -1 : // asc
            o1.getPrice() > o2.getPrice() ? 1 :
                ORDER_COMPARATOR.compare(o1, o2);

    private final AtomicBoolean taskScheduled = new AtomicBoolean(false);
    private final Map<Long, E> ordersByIndex = new HashMap<>();
    private final OrderSet<E> buyOrders = new OrderSet<>(BUY_COMPARATOR);
    private final OrderSet<E> sellOrders = new OrderSet<>(SELL_COMPARATOR);
    private final IndexedTxModel<E> txModel;
    private final Executor executor;
    private DxTimer.Cancellable task;
    private MarketDepthListener<E> listener;
    private long aggregationPeriodMillis;
    private int depthLimit;

    private MarketDepthModel(Builder<E> builder) {
        depthLimit = builder.depthLimit;
        buyOrders.setDepthLimit(depthLimit);
        sellOrders.setDepthLimit(depthLimit);
        listener = builder.listener;
        // TODO: replace to public api from DXEndpoint
        executor = builder.executor != null
            ? builder.executor
            : ((DXFeedImpl) builder.feed).getDXEndpoint().getOrCreateExecutor();
        aggregationPeriodMillis = builder.aggregationPeriodMillis;
        txModel = builder.txModelBuilder.withListener(this::eventsReceived).build();
    }

    public static <E extends OrderBase> Builder<E> newBuilder(Class<E> eventType) {
        return new Builder<>(eventType);
    }

    public void attachFeed(DXFeed feed) {
        txModel.attach(feed);
    }

    public void detachFeed(DXFeed feed) {
        txModel.detach(feed);
    }

    public synchronized int getDepthLimit() {
        return depthLimit;
    }

    public synchronized void setDepthLimit(int depthLimit) {
        if (depthLimit < 0)
            depthLimit = 0;
        if (this.depthLimit == depthLimit)
            return;
        this.depthLimit = depthLimit;
        buyOrders.setDepthLimit(depthLimit);
        sellOrders.setDepthLimit(depthLimit);
        // TODO: call in executor
        tryCancelTask();
        notifyListeners();
    }

    public synchronized long getAggregationPeriod() {
        return aggregationPeriodMillis;
    }

    public synchronized void setAggregationPeriod(long aggregationPeriod, TimeUnit unit) {
        long aggregationPeriodMillis = unit.toMillis(aggregationPeriod);
        if (aggregationPeriodMillis < 0)
            aggregationPeriodMillis = 0;
        if (this.aggregationPeriodMillis == aggregationPeriodMillis)
            return;
        this.aggregationPeriodMillis = aggregationPeriodMillis;
        rescheduleTaskIfNeeded(aggregationPeriodMillis);
    }

    @Override
    public void close() {
        txModel.close();
        tryCancelTask();
        listener = null;
    }

    // only for testing
    void processEvents(List<E> events) {
        txModel.processEvents(events);
    }

    private synchronized void eventsReceived(IndexedEventSource source, List<E> events, boolean isSnapshot) {
        if (update(source, events, isSnapshot)) {
            if (isSnapshot || aggregationPeriodMillis == 0) {
                tryCancelTask();
                notifyListeners();
            } else {
                scheduleTaskIfNeeded(aggregationPeriodMillis);
            }
        }
    }

    private synchronized void notifyListeners() {
        try {
            MarketDepthListener<E> listener = this.listener;
            if (listener == null)
                return;
            listener.modelChanged(new MarketDepthListener.OrderBook<>(getBuyOrders(), getSellOrders()));
        } finally {
            taskScheduled.set(false);
        }
    }

    private synchronized void scheduleTaskIfNeeded(long delayMillis) {
        if (taskScheduled.compareAndSet(false, true)) {
            task = DxTimer.getInstance().runOnce(() -> executor.execute(this::notifyListeners), delayMillis);
        }
    }

    private synchronized void rescheduleTaskIfNeeded(long delayMillis) {
        if (tryCancelTask() && delayMillis != 0) {
            scheduleTaskIfNeeded(delayMillis);
        }
    }

    private synchronized boolean tryCancelTask() {
        if (taskScheduled.get() && task != null ) {
            task.cancel();
            taskScheduled.set(false);
            return true;
        }
        return false;
    }

    private boolean update(IndexedEventSource source, List<E> events, boolean isSnapshot) {
        if (isSnapshot) {
            clearBySource(source);
        }
        for (E order : events) {
            E removed = ordersByIndex.remove(order.getIndex());
            if (removed != null) {
                getOrderSetForOrder(removed).remove(removed);
            }
            if (shallAdd(order)) {
                ordersByIndex.put(order.getIndex(), order);
                getOrderSetForOrder(order).add(order);
            }
        }
        return isChanged();
    }

    private boolean isChanged() {
        return buyOrders.isChanged() || sellOrders.isChanged();
    }

    private List<E> getBuyOrders() {
        return buyOrders.toList();
    }

    private List<E> getSellOrders() {
        return sellOrders.toList();
    }

    private void clearBySource(IndexedEventSource source) {
        ordersByIndex.entrySet().removeIf(entry -> entry.getValue().getSource().equals(source));
        buyOrders.clearBySource(source);
        sellOrders.clearBySource(source);
    }

    private boolean shallAdd(E order) {
        return order.hasSize() && (order.getEventFlags() & IndexedEvent.REMOVE_EVENT) == 0;
    }

    private OrderSet<E> getOrderSetForOrder(E order) {
        return (order.getOrderSide() == Side.BUY) ? buyOrders : sellOrders;
    }

    private static int compareString(String s1, String s2) {
        return (s1 != null) ? ((s2 != null) ? s1.compareTo(s2) : 1) : ((s2 != null) ? -1 : 0);
    }

    public static class Builder<E extends OrderBase> {
        private final IndexedTxModel.Builder<E> txModelBuilder;
        private DXFeed feed;
        private Executor executor;
        private MarketDepthListener<E> listener;
        private long aggregationPeriodMillis;
        private int depthLimit;

        public Builder(Class<E> eventType) {
            txModelBuilder = IndexedTxModel.newBuilder(eventType);
        }

        public Builder<E> withFeed(DXFeed feed) {
            txModelBuilder.withFeed(feed);
            this.feed = feed;
            return this;
        }

        public Builder<E> withSources(OrderSource... sources) {
            txModelBuilder.withSources(sources);
            return this;
        }

        public Builder<E> withSources(Collection<OrderSource> sources) {
            txModelBuilder.withSources(sources);
            return this;
        }

        public Builder<E> withSymbol(String symbol) {
            txModelBuilder.withSymbol(symbol);
            return this;
        }

        public Builder<E> withExecutor(Executor executor) {
            txModelBuilder.withExecutor(executor);
            this.executor = executor;
            return this;
        }

        public Builder<E> withListener(MarketDepthListener<E> listener) {
            this.listener = listener;
            return this;
        }

        public Builder<E> withAggregationPeriod(int aggregationPeriod, TimeUnit unit) {
            long aggregationPeriodMillis = unit.toMillis(aggregationPeriod);
            if (aggregationPeriodMillis < 0)
                aggregationPeriodMillis = 0;
            this.aggregationPeriodMillis = aggregationPeriodMillis;
            return this;
        }

        public Builder<E> withDepthLimit(int depthLimit) {
            if (depthLimit < 0)
                depthLimit = 0;
            this.depthLimit = depthLimit;
            return this;
        }

        public MarketDepthModel<E> build() {
            return new MarketDepthModel<>(this);
        }
    }

    private static class OrderSet<E extends OrderBase> {
        private final List<E> snapshot = new ArrayList<>();
        private final Comparator<? super OrderBase> comparator;
        private final TreeSet<E> orders;
        private int depthLimit;
        private boolean isChanged;

        public OrderSet(Comparator<? super OrderBase> comparator) {
            this.comparator = comparator;
            orders = new TreeSet<>(comparator);
        }

        public void setDepthLimit(int depthLimit) {
            if (this.depthLimit == depthLimit)
                return;
            this.depthLimit = depthLimit;
            isChanged = true;
        }

        public void add(E order) {
            if (orders.add(order))
                markAsChangedIfNeeded(order);
        }

        public void remove(E order) {
            if (orders.remove(order))
                markAsChangedIfNeeded(order);
        }

        public boolean isChanged() {
            return isChanged;
        }

        public void clearBySource(IndexedEventSource source) {
            isChanged = orders.removeIf(order -> order.getSource().equals(source));
        }

        public List<E> toList() {
            if (isChanged)
                updateSnapshot();
            return new ArrayList<>(snapshot);
        }

        private void updateSnapshot() {
            isChanged = false;
            snapshot.clear();
            int limit = isDepthLimitUnbounded() ? Integer.MAX_VALUE : depthLimit;
            Iterator<E> it = orders.iterator();
            for (int i = 0; i < limit && it.hasNext(); ++i) {
                snapshot.add(it.next());
            }
        }

        private void markAsChangedIfNeeded(E order) {
            if (isChanged)
                return;
            if (isDepthLimitUnbounded() || isSizeWithinDepthLimit() || isOrderWithinDepthLimit(order))
                isChanged = true;
        }

        private boolean isDepthLimitUnbounded() {
            return depthLimit <= 0 || depthLimit == Integer.MAX_VALUE;
        }

        private boolean isSizeWithinDepthLimit() {
            return orders.size() <= depthLimit;
        }

        private boolean isOrderWithinDepthLimit(E order) {
            if (snapshot.isEmpty())
                return true;
            E last = snapshot.get(snapshot.size() - 1);
            return comparator.compare(last, order) >= 0;
        }
    }
}
