package com.devexperts.mdd.news.model;

import com.devexperts.mdd.news.event.NewsEvent;
import com.devexperts.mdd.news.event.NewsFilter;
import com.devexperts.mdd.news.event.NewsKey;

import java.io.Serializable;
import java.util.List;

/**
 * News list with the last available news key, before and after keys. Last news key is used to not receive duplicate news.
 * Before and after keys use for pagination search
 * @see NewsService#findNewsForFilter(NewsFilter, NewsKey)
 */
public class NewsList implements Serializable {
    
    private static final long serialVersionUID = 2L;

    private final List<NewsEvent> news;
    private final NewsKey lastKey;
    private final NewsKey afterKey;
    private final NewsKey beforeKey;
    private NewsSource source = null;
    private Boolean isPolling = null;

    public NewsList(List<NewsEvent> news, NewsKey lastKey) {
        this.news = news;
        this.lastKey = lastKey;
        this.afterKey = news.isEmpty() ? lastKey : news.get(0).getKey();
        this.beforeKey = news.isEmpty() ? lastKey : news.get(news.size() - 1).getKey();
    }

    /**
     * Returns list of news.
     * @return list of news (summary with a news body).
     */
    public List<NewsEvent> getNews() {
        return news;
    }

    /**
     * Returns last news key for which the news list is valid.
     * @return last news key
     */
    public NewsKey getLastKey() {
        return lastKey;
    }

    /**
     * Returns news key which can be use for pagination search.
     * @see com.devexperts.mdd.news.event.Direction#AFTER
     * @return after news key
     */
    public NewsKey getAfterKey() {
        return afterKey;
    }

    /**
     * Returns news key which can be use for pagination search.
     * @see com.devexperts.mdd.news.event.Direction#BEFORE
     * @return before news key
     */
    public NewsKey getBeforeKey() {
        return beforeKey;
    }

    /**
     * Set flag is all news list was taken from cache.
     */
    public void setSource(NewsSource source) {
        this.source = source;
    }

    /**
     * Set flag is news list filled by polling.
     */
    public void setIsPolling(boolean isPolling) {
        this.isPolling = isPolling;
    }


    /**
     * Returns is all news list was taken from cache.
     * @return isCache
     */
    public NewsSource getSource() {
        return source;
    }

    /**
     * Returns is news list filled by polling.
     * @return isPolling
     */
    public Boolean isPolling() {
        return isPolling;
    }
}
