package com.devexperts.mdd.news.event;

import com.devexperts.util.TimeFormat;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Contains news details including time, title, news source, etc.
 */
public class NewsEvent implements Serializable, Comparable<NewsEvent> {

    private static final long serialVersionUID = 3L;

    /** Symbol used for QD Stream Subscription. */
    public static final String MESSAGE_SYMBOL = "NEWS";

    // News identifier
    private final NewsKey key;

    private final NewsKey originalKey;
    private final NewsKey chainKey;

    // Required fields
    private final String feed;
    private final String sourceId;
    private final String sourcePrefix;
    private final String source;
    private final String sourceUrl;
    private final long time;
    private final String title;
    private transient NewsOrigin origin;

    // Optional fields (may be null/empty)
    private final String body;
    private final String extraAttributes;
    private final String action;
    private final String actionReason;
    private final String newsContent;
    private final List<String> symbols;
    private final Map<String, List<String>> tags;

    protected NewsEvent(Builder builder) {
        this.key = Objects.requireNonNull(builder.key, "key");

        // Non-null fields
        this.feed = notNull(NewsTags.NEWS_FEED, builder.feed);
        this.sourceId = notNull(NewsTags.NEWS_SOURCE_ID, builder.sourceId);
        this.source = notNull(NewsTags.NEWS_SOURCE, builder.source);
        this.sourcePrefix = builder.sourcePrefix;
        this.origin = NewsOrigin.valueOf(feed, sourcePrefix, source);
        this.title = notNull(NewsTags.NEWS_TITLE, builder.title);
        // Non-null time
        this.time = builder.time;
        if (this.time <= 0)
            throw new IllegalArgumentException("Illegal '" + NewsTags.NEWS_TIME + "': invalid time");

        // Optional 
        this.body = (builder.body == null || builder.body.isEmpty()) ? null : builder.body;
        this.extraAttributes = (builder.extraAttributes == null || builder.extraAttributes.isEmpty()) ? null : builder.extraAttributes;
        this.action = (builder.action == null || builder.action.isEmpty()) ? null : builder.action;
        this.sourceUrl = (builder.sourceUrl == null || builder.sourceUrl.isEmpty()) ? null : builder.sourceUrl;
        this.actionReason = (builder.actionReason == null || builder.actionReason.isEmpty()) ? null : builder.actionReason;
        this.newsContent = (builder.newsContent == null || builder.newsContent.isEmpty()) ? null : builder.newsContent;
        this.originalKey = builder.originalKey;
        this.chainKey = builder.chainKey;
        this.symbols = immutableNonEmptyList(builder.symbols);
        this.tags = immutableMap(builder.tags);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    /** Returns news key for identifying news. */
    public NewsKey getKey() {
        return key;
    }

    public NewsKey getOriginalKey() {
        return originalKey;
    }

    public NewsKey getChainKey() {
        return chainKey;
    }

    /**
     * Returns original news key if event is correction or else news id (originalKey for non-corrected news is null) .
     */
    public NewsKey getOriginalKeyOrSelf() {
        return originalKey != null ? originalKey : key;
    }

    /** Returns news feed ID. */
    public String getFeed() {
        return feed;
    }

    /** Returns external ID assigned by news provider. */
    public String getSourceId() {
        return sourceId;
    }

    /** Returns news source provider. */
    public String getSource() {
        return source;
    }

    /** Returns news source url. */
    public String getSourceUrl() {
        return sourceUrl;
    }

    public String getSourcePrefix() {
        return sourcePrefix;
    }

    /** Return news origin (feed and source provider) */
    public NewsOrigin getOrigin() {
        return origin;
    }

    /** Returns news time. */
    public long getTime() {
        return time;
    }

    /** Returns news title. */
    public String getTitle() {
        return title;
    }

    /** Returns news body, or {@code null}. */
    public String getBody() {
        return body;
    }

    /** Returns news extra attributes, or {@code null}. */
    public String getExtraAttributes() {
        return extraAttributes;
    }

    /** Returns news action, or {@code null}. */
    public String getAction() {
        return action;
    }

    /** Returns news action reason, or {@code null}. */
    public String getActionReason() {
        return actionReason;
    }

    public String getNewsContent() {
        return newsContent;
    }

    /** Returns set of instrument symbols for this news, or {@code null} if not present. */
    public List<String> getSymbols() {
        return symbols;
    }

    /**
     * Returns set of custom tags available for the news. This set is never {@code null} but can be empty.
     */
    public Map<String, List<String>> getTags() {
        return tags;
    }

    /**
     * Events are naturally ordered by their arrival time to the news storage.
     */
    public int compareTo(NewsEvent o) {
        return getKey().getCode().compareTo(o.getKey().getCode());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        } else if (o instanceof NewsEvent) {
            NewsEvent that = (NewsEvent)o;
            return this.getKey().equals(that.getKey());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return getKey().hashCode();
    }

    @Override
    public String toString() {
        return "NewsEvent{" + key.getCode() + "/" + getSourceId() +
            ", time=" + TimeFormat.DEFAULT.format(getTime()) +
            ", feed=" + feed + ", title=" + title +
            "}";
    }

    private static String notNull(String name, String value) {
        if (value == null || value.isEmpty())
            throw new IllegalArgumentException("Illegal '" + name + "': empty or null value");
        return value;
    }

    private static List<String> immutableNonEmptyList(List<String> values) {
        if (values == null || values.isEmpty())
            return null;
        return Collections.unmodifiableList(new ArrayList<>(values));
    }

    private static Map<String, List<String>> immutableMap(Map<String, List<String>> map) {
        if (map == null || map.isEmpty())
            return Collections.emptyMap();
        LinkedHashMap<String, List<String>> copy = new LinkedHashMap<>(map);
        copy.replaceAll((k, values) -> Collections.unmodifiableList(new ArrayList<>(values)));
        return Collections.unmodifiableMap(copy);
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.origin = NewsOrigin.valueOf(feed, sourcePrefix, source);
    }

    // Builder class

    /**
     * Builder class for {@link NewsEvent}.
     */
    public static class Builder {
        private NewsKey key;
        private NewsKey originalKey;
        private NewsKey chainKey;
        private String feed;
        private String sourceId;
        private String sourcePrefix;
        private String source;
        private String sourceUrl;
        private long time;
        private String title;
        private String body;
        private String extraAttributes;
        private String action;
        private String actionReason;
        private String newsContent;
        private List<String> symbols;
        private final Map<String, List<String>> tags = new LinkedHashMap<>();

        protected Builder() {}

        public Builder withKey(NewsKey key) {
            this.key = key;
            return this;
        }

        public Builder withOriginalKey(NewsKey originalKey) {
            this.originalKey = originalKey;
            return this;
        }

        public Builder withChainKey(NewsKey chainKey) {
            this.chainKey = chainKey;
            return this;
        }

        public Builder withFeed(String feed) {
            this.feed = feed;
            return this;
        }

        public Builder withSourceId(String sourceId) {
            this.sourceId = sourceId;
            return this;
        }

        public Builder withSourcePrefix(String sourcePrefix) {
            this.sourcePrefix = sourcePrefix;
            return this;
        }

        public Builder withSource(String source) {
            this.source = source;
            return this;
        }

        public Builder withSourceUrl(String sourceUrl) {
            this.sourceUrl = sourceUrl;
            return this;
        }

        public Builder withTime(long time) {
            this.time = time;
            return this;
        }

        public Builder withTitle(String title) {
            this.title = title;
            return this;
        }

        public Builder withBody(String body) {
            this.body = body;
            return this;
        }

        public Builder withExtraAttributes(String extraAttributes) {
            this.extraAttributes = extraAttributes;
            return this;
        }

        public Builder withAction(String action) {
            this.action = action;
            return this;
        }

        public Builder withActionReason(String actionReason) {
            this.actionReason = actionReason;
            return this;
        }

        public Builder withNewsContent(String newsContent) {
            this.newsContent = newsContent;
            return this;
        }

        public Builder withSymbols(List<String> symbols) {
            if (symbols != null && !symbols.isEmpty()) {
                this.symbols = new ArrayList<>(symbols);
            } else {
                this.symbols = null;
            }
            return this;
        }

        public Builder withTags(Map<String, List<String>> values) {
            if (values != null && !values.isEmpty()) {
                values.forEach(this::withTag);
            }
            return this;
        }

        public Builder withTag(String tag, List<String> values) {
            if (tag != null && !tag.isEmpty()) {
                if (values != null && !values.isEmpty()) {
                    tags.put(tag, new ArrayList<>(values));
                } else {
                    tags.remove(tag);
                }
            }
            return this;
        }

        public Builder addTag(String tag, String value) {
            if (tag != null && !tag.isEmpty() && value != null && !value.isEmpty()) {
                tags.computeIfAbsent(tag, k -> new ArrayList<>()).add(value);
            }
            return this;
        }

        public NewsEvent build() {
            NewsEvent event = new NewsEvent(this);
            // Reset state
            return event;
        }
    }
}
