package com.devexperts.mdd.news.event;

import java.io.Serializable;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/**
 * This class describe pair of "news feed" and "news source", where "news source"
 * is subdivision of feed, as each feed provide {@link NewsEvent NewsEvents}
 * from one or more sources.
 * <p>
 * Origin can be exact, with fixed feed and source, and can be a mask, where
 * feed, source or both are set to '*'.
 * <p>
 * Each {@link NewsEvent NewsEvent's} origin is always exact.
 * <p>
 * Origins with masks can be used as {@link Predicate<NewsEvent> filters} for
 * {@link NewsEvent NewsEvents}.
 * <p>
 * String representation of origin is {@code &lt;feed&gt;:&lt;source&gt;}, where {@code &lt;feed&gt;}
 * is alphanumeric string which starts from letter and {@code &lt;source&gt;} is arbitrary string
 * of printable characters.
 * <p>
 * When {@link NewsOrigin} is created, empty {@code &lt;feed&gt;} or {@code &lt;source&gt;} is interpreted
 * as {@code *} and string without {@code :} (semicolon) is interpreted as {@code &lt;feed&gt;:*}.
 * <p>
 * Empty string is special value that means "nothing".
 */
public final class NewsOrigin implements Serializable {
    /**
     * {@link NewsOrigin} which matches any feed and source, {@code *:*}.
     */
    public final static NewsOrigin ANY = new NewsOrigin("*", "*", "*");

    /**
     * {@link  NewsOrigin} which matches with nothing.
     */
    public final static NewsOrigin NOTHING = new NewsOrigin();

    private final String feed;
    private final String sourcePrefix;
    private final String source;

    public static NewsOrigin valueOf(String origin) {
        Objects.requireNonNull(origin, "origin");
        if (origin.isEmpty())
            return NOTHING;
        String[] originParts = origin.split(":");

        // String without semicolon is feed only (like feed:*:*)
        // String with only one semicolon is for source without prefix (like feed:source)
        String feed = originParts[0];
        String sourcePrefix = originParts.length == 3 ? originParts[1] : "*";
        String source = originParts.length > 1 ? originParts[originParts.length - 1] : "*";
        return valueOf(feed, sourcePrefix, source);
    }

    public static NewsOrigin valueOf(String feed, String sourcePrefix, String source) {
        checkFeedFormat(feed);
        Objects.requireNonNull(source, "source");
        if (sourcePrefix == null) {
            sourcePrefix = "";
        }
        if (feed.isEmpty() && source.isEmpty() && sourcePrefix.isEmpty())
            return NOTHING;
        if (("*".equals(feed) || feed.isEmpty()) && ("*".equals(source) || source.isEmpty()) && "*".equals(sourcePrefix))
            return ANY;

        return new NewsOrigin(feed.isEmpty() ? "*" : feed, sourcePrefix, source.isEmpty() ? "*" : source);
    }

    public static NewsOrigin valueOf(NewsEvent event) {
        Objects.requireNonNull(event, "event");
        return new NewsOrigin(event.getFeed(), event.getSourcePrefix(), event.getSource());
    }

    private NewsOrigin(String feed, String sourcePrefix, String source) {
        this.feed = feed;
        this.sourcePrefix = sourcePrefix;
        this.source = source;
    }

    private NewsOrigin() {
        this.feed = "";
        this.sourcePrefix = "";
        this.source = "";
    }

    public String getFeed() {
        return feed;
    }

    public String getSource() {
        return source;
    }

    public String getSourcePrefix() {
        return sourcePrefix;
    }

    public boolean isFeedWildcard() {
        return "*".equals(feed);
    }

    public boolean isSourceWildcard() {
        return "*".equals(source);
    }

    public boolean isSourcePrefixWildcard() {
        return "*".equals(sourcePrefix);
    }

    public boolean isWildcard() {
        return isFeedWildcard() && isSourceWildcard() && isSourcePrefixWildcard();
    }

    public boolean isExact() {
        return !isFeedWildcard() && !isSourceWildcard() && !feed.isEmpty() && !source.isEmpty() && (sourcePrefix == null || !isSourcePrefixWildcard());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        NewsOrigin that = (NewsOrigin) o;
        return Objects.equals(feed, that.feed) && Objects.equals(sourcePrefix, that.sourcePrefix)
            && Objects.equals(source, that.source);
    }

    @Override
    public int hashCode() {
        return Objects.hash(feed, sourcePrefix, source);
    }

    @Override
    public String toString() {
        return (feed.isEmpty() && source.isEmpty() && sourcePrefix.isEmpty()) ? "" :
            feed + ":" + sourcePrefix + ":" + source;
    }

    private final static Pattern FEED_PATTERN = Pattern.compile("^[a-zA-Z][a-zA-Z0-9]*$");
    private static void checkFeedFormat(String feed) {
        Objects.requireNonNull(feed, "feed");
        if (feed.isEmpty() || "*".equals(feed))
            return;
        if (FEED_PATTERN.matcher(feed).matches())
            return;
        throw new IllegalArgumentException("Invalid feed name \"" + feed + "\"");
    }

    public boolean isEmptySourcePrefix() {
        return sourcePrefix == null || sourcePrefix.isEmpty();
    }
}
