package com.dxfeed.orcs;

import com.devexperts.logging.Logging;
import com.devexperts.util.TimeFormat;
import com.devexperts.util.TimeUtil;
import com.dxfeed.event.market.Order;
import com.dxfeed.event.market.Side;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * Utility class to check consistency of {@link Order} list in terms that in each time bid price is less than ask price.
 * There is additional information may be gathered during the check:
 * <ul>
 *     <li>Gap detection for subsequent events. In other words if there was a pause between orders greater than {@code timeGapBound}</li>
 *     <li>Last bid and ask change with the minimal period of 1 second</li>
 *     <li>Spike in quotes</li>
 * </ul>
 *
 * <p>
 * Important to mention that sometimes it's a valid situation when bid greater or equal to ask.
 */
public class PriceLevelChecker {
    private static final Logging log = Logging.getLogging(PriceLevelChecker.class);

    private PriceLevelChecker() {
    }

    public static boolean validate(List<Order> orders, long timeGapBound, boolean printQuotes) {
        if (orders.isEmpty())
            return true;

        TreeMap<Double, Order> currentState = new TreeMap<>();
        List<Order> curTimeOrders = new ArrayList<>();
        long curTime = Long.MIN_VALUE;
        long lastBboTime = Long.MIN_VALUE;
        double prevBid = Double.NaN;
        double prevAsk = Double.NaN;
        boolean valid = true;
        long timeGap = 0;
        for (int i = 0; i < orders.size(); i++) {
            Order order = orders.get(i);
            if (i != 0) {
                Order prevOrder = orders.get(i - 1);
                long curTimeGap = order.getTime() - prevOrder.getTime();
                if (curTimeGap > timeGapBound) {
                    log.warn("TIME_GAP_BOUND between: " + prevOrder + " - " + order);
                }
                timeGap = Math.max(timeGap, curTimeGap);
            }

            if (order.getTime() != curTime) {
                // build bbo with 1sec granularity
                double bid = Double.NaN;
                double ask = Double.NaN;
                for (Order curOrder : currentState.values()) {
                    if (curOrder.getOrderSide() == Side.BUY && (Double.isNaN(bid) || bid < curOrder.getPrice())) {
                        bid = curOrder.getPrice();
                    }
                    if (curOrder.getOrderSide() == Side.SELL && (Double.isNaN(ask) || ask > curOrder.getPrice())) {
                        ask = curOrder.getPrice();
                    }
                }

                if (lastBboTime / TimeUtil.SECOND != curTime / TimeUtil.SECOND) {
                    if (!Double.isNaN(prevAsk) && !Double.isNaN(prevBid) && !Double.isNaN(ask) && !Double.isNaN(bid)) {
                        double bidSpike = (bid - prevBid) / prevBid;
                        double askSpike = (ask - prevAsk) / prevAsk;
                        if (printQuotes && (bidSpike >= 0.005 || askSpike >= 0.005)) {
                            log.warn("SPIKE: " + TimeFormat.DEFAULT.format(lastBboTime) + "=" + prevBid + "-" +
                                prevAsk + TimeFormat.DEFAULT.format(curTime) + "=" + bid + "-" + ask);
                        }
                    }
                    prevBid = bid;
                    prevAsk = ask;
                    lastBboTime = curTime;
                    if (printQuotes) {
                        log.info("time=" + TimeFormat.GMT.format(lastBboTime) + ", bid=" + prevBid + ", ask=" + prevAsk);
                    }
                }
                //do all checks here
                for (Order curOrder : curTimeOrders) {
                    if (curOrder.getOrderSide() == Side.BUY) {
                        Map.Entry<Double, Order> prevEntry = currentState.lowerEntry(curOrder.getPrice());
                        while (prevEntry != null) {
                            Order prevOrder = prevEntry.getValue();
                            if (prevOrder.getOrderSide() == Side.BUY)
                                break;
                            log.warn("CROSS: New curOrder: " + curOrder + " and prev " + prevOrder);
                            valid = false;
                            prevEntry = currentState.lowerEntry(prevOrder.getPrice());
                        }
                    } else if (curOrder.getOrderSide() == Side.SELL) {
                        Map.Entry<Double, Order> nextEntry = currentState.higherEntry(curOrder.getPrice());
                        while (nextEntry != null) {
                            Order nextOrder = nextEntry.getValue();
                            if (nextOrder.getOrderSide() == Side.SELL)
                                break;
                            log.warn("CROSS: New curOrder: " + curOrder + " and prev " + nextOrder);
                            valid = false;
                            nextEntry = currentState.higherEntry(nextOrder.getPrice());
                        }
                    }
                }
                curTime = order.getTime();
                curTimeOrders.clear();
            }
            curTimeOrders.add(order);
            if (order.getSize() != 0)
                currentState.put(order.getPrice(), order);
            else
                currentState.remove(order.getPrice());
        }
        if (timeGap > timeGapBound) {
            log.warn("TIME_GAP_BOUND: " + timeGap);
            valid = false;
        }
        return valid;
    }
}
