package com.dxfeed.orcs.api;

import com.dxfeed.event.candle.CandleSymbol;
import com.dxfeed.event.market.Order;
import com.dxfeed.event.market.OrderSource;
import com.dxfeed.event.market.Quote;
import com.dxfeed.orcs.qd.PriceLevelCandle;

import java.util.List;

public interface PriceLevelService {
    /**
     * Custom {@link CandleSymbol} attribute specifying granularity of price level updates in response
     */
    public static final String GRANULARITY_ATTRIBUTE_KEY = "gr";

    /**
     * Returns list of price levels for the specified {@link CandleSymbol} within passed {@code from} and {@code} to times.
     * The events are ordered by {@link Order#getTime() time} in the collection.
     * <p>
     * Passed candle symbol shall contain supported {@link com.dxfeed.event.candle.CandlePeriod} and custom attribute
     * called granularity with the key {@code GRANULARITY_ATTRIBUTE_KEY}. Granularity value shall be represented by
     * integer value in seconds. Minimal value for granularity is 1 second.
     * <p>
     * If passed {@link com.dxfeed.event.candle.CandlePeriod} or granularity value are not supported by the service
     * the empty list will be returned.
     *
     * @param candleSymbol {@link CandleSymbol} to request
     * @param sourceId     sourceId to request, {@link OrderSource#id()}
     * @param from         from time in UTC
     * @param to           to time in UTC
     * @param caller       caller identifier
     * @return list of {@link Order} events sorted in ascending order by time
     */
    public List<Order> getOrders(CandleSymbol candleSymbol, int sourceId, long from, long to, String caller);

    /**
     * Returns available to the client order sources and symbols for each {@link OrderSource}. Order source and symbols
     * are filtered according to the client permissions. Symbols and order sources view is built as of now, e.g.
     * the response contains only existing data (for example, no symbols that were delisted)
     *
     * @return {@link AuthOrderSource}
     */
    public AuthOrderSource getAuthOrderSource();

    /**
     * Returns list of quotes for the specified {@link CandleSymbol} within passed {@code from} and {@code} to times.
     * The quotes are ordered by {@link Quote#getTime() time} in the collection.
     *
     * @param candleSymbol the candle symbol to request (see {@link CandleSymbol})
     * @param from         from time in UTC
     * @param to           to time in UTC
     * @param caller       caller identifier
     * @return list of {@link Quote} for particular period of time
     */
    public List<Quote> getQuotes(CandleSymbol candleSymbol, long from, long to, String caller);

    /**
     * Returns list of price level candles for the specified {@link CandleSymbol} and date.
     * The candles are ordered by {@link PriceLevelCandle#getTime() time} in the collection.
     *
     * @param candleSymbol the candle symbol to request (see {@link CandleSymbol})
     * @param sourceId     sourceId, for example {@link OrderSource#id()}
     * @param ymd          date in yyyymmdd format
     * @param caller       caller identifier
     * @return list of {@link PriceLevelCandle} for particular symbol and day
     * @deprecated use {@link #getOrders(CandleSymbol, int, long, long, String)}
     */
    @Deprecated
    public List<PriceLevelCandle> getCandles(CandleSymbol candleSymbol, int sourceId, int ymd, String caller);

    /**
     * Returns list of price level candles for the specified {@link CandleSymbol} within passed {@code from} and {@code} to times.
     * The candles are ordered by {@link PriceLevelCandle#getTime() time} in the collection.
     *
     * @param candleSymbol the candle symbol to request (see {@link CandleSymbol})
     * @param sourceId     sourceId, for example {@link OrderSource#id()}
     * @param from         from time in UTC
     * @param to           to time in UTC
     * @param caller       caller identifier
     * @return list of {@link PriceLevelCandle} for particular period of time
     * @deprecated use {@link #getOrders(CandleSymbol, int, long, long, String)}
     */
    @Deprecated
    public List<PriceLevelCandle> getCandles(CandleSymbol candleSymbol, int sourceId, long from, long to, String caller);
}
