/*
 * Decompiled with CFR 0.152.
 */
package com.dxfeed.sample.api;

import com.devexperts.util.TimeFormat;
import com.dxfeed.api.DXEndpoint;
import com.dxfeed.api.DXFeed;
import com.dxfeed.api.DXFeedSubscription;
import com.dxfeed.event.market.Order;
import com.dxfeed.event.market.Scope;
import com.dxfeed.event.market.Side;
import com.dxfeed.event.market.TimeAndSale;
import com.dxfeed.ipf.InstrumentProfile;
import com.dxfeed.ipf.InstrumentProfileReader;
import com.dxfeed.schedule.Schedule;
import com.dxfeed.schedule.Session;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class Slicer
implements PropertyChangeListener {
    public static final int PERIODS_PER_BATCH = 100;
    public static final int MIN_BATCH_MILLIS = 120000;
    private static final ExecutorService EXECUTOR = Executors.newFixedThreadPool(4);
    protected ArrayList<String> symbols;
    protected long period;
    protected long startTime;
    protected long endTime;
    protected long oldThreshold;
    protected long waitThreshold;
    protected long sessionStartTime;
    private DXEndpoint orderEndpoint;
    private DXEndpoint saleEndpoint;
    private final CountDownLatch completionLatch = new CountDownLatch(2);
    private final Map<String, List<Slice>> allSlices = new HashMap<String, List<Slice>>();
    private long oldestActiveSliceTime;
    private long lastOrderTime;
    private long lastSaleTime;
    private long doneSlices;
    private int orderCount = 0;
    private int saleCount = 0;
    private boolean isRTOrTapeMode;
    private boolean zeroFillFromStartTimeToFirstTick = false;
    private boolean calcAvgBookPrices = false;
    private boolean useCompositeOrderOrQuote = false;
    private long[] avgBookSizes = new long[5];
    private String scheduleDef;
    private Schedule schedule = null;
    private BufferedWriter outputWriter;

    public static void main(String[] args) throws InterruptedException, IOException {
        new Slicer().start(args);
    }

    protected void start(String[] args) throws InterruptedException, IOException {
        Properties properties;
        String mode;
        if (args.length == 0 || args.length > 2) {
            this.showHelp();
            System.exit(-1);
        }
        if (!((mode = args[0].toLowerCase()).equals("realtime") || mode.equals("tape") || mode.equals("history"))) {
            this.showHelp();
            System.exit(-1);
        }
        String configFile = "slicer.cfg";
        if (args.length == 2) {
            configFile = args[1];
        }
        if ((properties = this.loadConfiguration(configFile)).getProperty("avgBookSizes", "").length() > 0) {
            this.calcAvgBookPrices = true;
            String[] bookSizes = properties.getProperty("avgBookSizes", "").split(",");
            if (bookSizes.length != 5) {
                System.out.println("Invalid value for avgBookSizes: must be a list of 5 comma-separated values");
                System.exit(-1);
            }
            for (int i = 0; i < 5; ++i) {
                this.avgBookSizes[i] = Long.parseLong(bookSizes[i]);
            }
        }
        this.scheduleDef = properties.getProperty("schedule", "");
        if (this.scheduleDef.length() > 0) {
            this.schedule = Schedule.getInstance((String)this.scheduleDef);
        }
        this.getSymbols(properties.getProperty("symbols", ""));
        this.period = (long)Integer.parseInt(properties.getProperty("period", "")) * 1000L;
        this.initOutputFile(properties.getProperty("outputFile", "slicer.cfg"));
        this.useCompositeOrderOrQuote = Boolean.parseBoolean(properties.getProperty("useCompositeOrderOrQuote", "false"));
        switch (mode) {
            case "realtime": {
                this.startTime = 0L;
                this.endTime = Long.MAX_VALUE;
                this.oldThreshold = Math.max(120000L, this.period * 5L);
                this.waitThreshold = Long.MAX_VALUE;
                this.isRTOrTapeMode = true;
                this.sessionStartTime = this.getSessionStartTime(System.currentTimeMillis());
                DXEndpoint endpoint = DXEndpoint.create((DXEndpoint.Role)DXEndpoint.Role.FEED).executor((Executor)EXECUTOR);
                this.startProcessing(endpoint.getFeed(), endpoint.getFeed());
                endpoint.connect(properties.getProperty("feedAddress", ""));
                this.awaitCompletionAndShowProgress();
                break;
            }
            case "tape": {
                this.startTime = TimeFormat.DEFAULT.parse(properties.getProperty("startTime", "")).getTime() / this.period * this.period;
                this.endTime = TimeFormat.DEFAULT.parse(properties.getProperty("endTime", "")).getTime() / this.period * this.period;
                this.oldThreshold = Math.max(120000L, this.period * 100L);
                this.waitThreshold = this.oldThreshold / 2L;
                this.isRTOrTapeMode = true;
                this.zeroFillFromStartTimeToFirstTick = Boolean.valueOf(properties.getProperty("zeroFillFromStartTimeToFirstTick", "false"));
                String tapeAddress = properties.getProperty("tapeAddress", "");
                this.readFiles(tapeAddress, tapeAddress);
                this.finish();
                break;
            }
            case "history": {
                this.startTime = TimeFormat.DEFAULT.parse(properties.getProperty("startTime", "")).getTime() / this.period * this.period;
                this.endTime = TimeFormat.DEFAULT.parse(properties.getProperty("endTime", "")).getTime() / this.period * this.period;
                this.oldThreshold = Math.max(120000L, this.period * 100L);
                this.waitThreshold = this.oldThreshold / 2L;
                this.isRTOrTapeMode = false;
                this.zeroFillFromStartTimeToFirstTick = Boolean.valueOf(properties.getProperty("zeroFillFromStartTimeToFirstTick", "false"));
                String orderAddress = properties.getProperty("historyOrderAddress", "");
                String saleAddress = properties.getProperty("historySaleAddress", "");
                this.readFiles(orderAddress, saleAddress);
                this.finish();
                break;
            }
            default: {
                this.showHelp();
            }
        }
        this.outputWriter.close();
        System.exit(0);
    }

    private Properties loadConfiguration(String configFile) {
        Properties properties = new Properties();
        File file = new File(configFile);
        if (!file.exists()) {
            this.showHelp();
        }
        if (file.exists()) {
            System.out.println("Loading configuration from " + file.getAbsoluteFile());
            try {
                properties.load(new FileInputStream(file));
                properties.list(System.out);
                System.out.println("");
            }
            catch (IOException e) {
                System.out.println("Failed to load configuration from " + file.getAbsoluteFile());
                System.exit(-1);
            }
        }
        return properties;
    }

    private void showHelp() {
        System.out.println("Usage: Slicer [realtime|tape|history] [config-file]");
        System.out.println("The following properties should be defined in the configuration file (defaults to slicer.cfg):");
        System.out.println("1) For real-time feed slicing: feed-address, symbols, period, [avgBookSizes], [outputFile], [schedule]");
        System.out.println("2) For tape slicing: tape-address, symbols, period, startTime, endTime, [size1, size2, size3, size4, size5], [outputFile], [schedule]");
        System.out.println("3) For history slicing:  order-address, sale-address, symbols, period, startTime, endTime, [size1, size2, size3, size4, size5], [outputFile], [schedule]");
        System.out.println("  feedAddress                       - dxFeed address for RT feed or file name");
        System.out.println("  tapeAddress                       - dxFeed tape file name. ~-notation is supported to list through local files with dates in the name");
        System.out.println("  historyOrderAddress,");
        System.out.println("  historySaleAddress                - dxFeed history file names for Order and TimeAndSale/TradeHistory events. ~-notation is supported to list through local files with dates in the name");
        System.out.println("  symbols                           - comma-separated list of symbols or IPF file name");
        System.out.println("  period                            - slice duration in seconds");
        System.out.println("  startTime                         - time of first slice, yyyyMMdd-HHmmssZ");
        System.out.println("  endTime                           - time of last slice, yyyyMMdd-HHmmssZ");
        System.out.println("  [avgBookSizes]                    - 5 comma-separated sizes to calculate average book prices upon; if omitted book average prices will not be calculated");
        System.out.println("  [outputFile]                      - output CSV file name; defaults to slicer.csv");
        System.out.println("  [schedule]                        - optional schedule to mark slices invalid during trading session pause");
        System.out.println("  [useCompositeOrderOrQuote]        - use composite orders or quote events instead of aggregate scope orders");
    }

    private void initOutputFile(String fileName) throws IOException {
        this.outputWriter = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(fileName), StandardCharsets.UTF_8));
        if (this.calcAvgBookPrices) {
            this.outputWriteLn("SYMBOL, TIME, BEST_BID, BEST_BID_SIZE, BEST_OFFER, BEST_OFFER_SIZE, AVG_BID_PRICE_1, AVG_OFFER_PRICE_1, AVG_BID_PRICE_2, AVG_OFFER_PRICE_2, AVG_BID_PRICE_3, AVG_OFFER_PRICE_3, AVG_BID_PRICE_4, AVG_OFFER_PRICE_4, AVG_BID_PRICE_5, AVG_OFFER_PRICE_5, VWAP, CUMULATIVE_VOLUME, IS_VALID_SLICE");
        } else {
            this.outputWriteLn("SYMBOL, TIME, BEST_BID, BEST_BID_SIZE, BEST_OFFER, BEST_OFFER_SIZE, VWAP, CUMULATIVE_VOLUME, IS_VALID_SLICE");
        }
    }

    private void outputWriteLn(String s) {
        try {
            this.outputWriter.write(s);
            this.outputWriter.newLine();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void outputFlush() {
        try {
            this.outputWriter.flush();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void getSymbols(String arg) throws IOException {
        if (arg.toLowerCase().contains(".ipf")) {
            System.out.print("Loading profiles from " + arg + "...");
            this.symbols = new ArrayList();
            for (InstrumentProfile profile : new InstrumentProfileReader().readFromFile(arg)) {
                this.symbols.add(profile.getSymbol());
            }
            System.out.println(this.symbols.size() + " profiles loaded");
        } else {
            this.symbols = new ArrayList<String>(Arrays.asList(arg.split(",")));
        }
    }

    private void finish() {
        this.checkOld();
        this.report("Finished");
    }

    protected void readFiles(String orderAddress, String saleAddress) throws InterruptedException {
        this.orderEndpoint = DXEndpoint.create((DXEndpoint.Role)DXEndpoint.Role.STREAM_FEED).executor((Executor)EXECUTOR);
        this.saleEndpoint = DXEndpoint.create((DXEndpoint.Role)DXEndpoint.Role.STREAM_FEED).executor((Executor)EXECUTOR);
        this.orderEndpoint.addStateChangeListener((PropertyChangeListener)this);
        this.saleEndpoint.addStateChangeListener((PropertyChangeListener)this);
        this.startProcessing(this.orderEndpoint.getFeed(), this.saleEndpoint.getFeed());
        this.sessionStartTime = this.getSessionStartTime(this.startTime);
        String props = "[start=" + TimeFormat.DEFAULT.format(this.sessionStartTime) + "][stop=" + TimeFormat.DEFAULT.format(this.endTime) + "][speed=max]";
        this.orderEndpoint.connect(orderAddress + props + "[name=order]");
        this.saleEndpoint.connect(saleAddress + props + "[name=sale]");
        this.awaitCompletionAndShowProgress();
    }

    private long getSessionStartTime(long startTime) {
        if (this.schedule == null) {
            return startTime - 86400000L;
        }
        Session startSession = this.schedule.getSessionByTime(startTime);
        return startSession.isTrading() ? startSession.getStartTime() : startTime;
    }

    private void awaitCompletionAndShowProgress() throws InterruptedException {
        while (this.completionLatch.getCount() > 0L) {
            this.report("Progress");
            this.completionLatch.await(10000L, TimeUnit.MILLISECONDS);
        }
    }

    private static String fmt(long time) {
        return time == Long.MAX_VALUE ? "done" : TimeFormat.DEFAULT.format(time);
    }

    private void startProcessing(DXFeed orderFeed, DXFeed saleFeed) {
        System.out.println("Starting: " + this.symbols.size() + " symbols, " + this.allSlices.size() + " sliced, " + this.doneSlices + " slices, startTime " + TimeFormat.DEFAULT.format(this.startTime) + ", endTime " + TimeFormat.DEFAULT.format(this.endTime));
        this.startProcessing(orderFeed, Order.class);
        this.startProcessing(saleFeed, TimeAndSale.class);
    }

    private void report(String what) {
        long avgSize = 0L;
        long maxSize = 0L;
        long firstSliceDate = 0L;
        long lastSliceDate = 0L;
        for (String symbol : this.symbols) {
            List<Slice> sl = this.allSlices.get(symbol);
            if (sl == null) continue;
            avgSize += (long)sl.size();
            if ((long)sl.size() <= maxSize) continue;
            maxSize = sl.size();
            firstSliceDate = sl.get((int)0).startTime;
            lastSliceDate = sl.get((int)(sl.size() - 1)).startTime;
        }
        System.out.println(Slicer.fmt(System.currentTimeMillis()) + ": " + what + ": " + this.symbols.size() + " symbols, " + this.allSlices.size() + " symbols sliced, " + this.doneSlices + " done slices, " + (avgSize /= (long)this.symbols.size()) + " avg. slices per symbol, " + maxSize + " max slices per symbol (" + Slicer.fmt(firstSliceDate) + " - " + Slicer.fmt(lastSliceDate) + "), lastOrderTime " + Slicer.fmt(this.lastOrderTime) + ", lastSaleTime " + Slicer.fmt(this.lastSaleTime) + ", oldestActiveSliceTime " + Slicer.fmt(this.oldestActiveSliceTime) + ", orderCount " + this.orderCount + ", saleCount " + this.saleCount);
    }

    private void startProcessing(DXFeed feed, Class<?> eventType) {
        DXFeedSubscription subscription = feed.createSubscription(eventType);
        subscription.addEventListener(events -> {
            try {
                this.process(eventType, events);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
        subscription.setSymbols(this.symbols);
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        try {
            this.checkCompletion(evt, this.orderEndpoint, true);
            this.checkCompletion(evt, this.saleEndpoint, false);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkCompletion(PropertyChangeEvent evt, DXEndpoint endpoint, boolean order) throws InterruptedException {
        if (evt.getSource() == endpoint && endpoint.getState() == DXEndpoint.State.NOT_CONNECTED) {
            endpoint.closeAndAwaitTermination();
            System.out.println("Completed processing " + (order ? "orders" : "sales"));
            Slicer slicer = this;
            synchronized (slicer) {
                if (order) {
                    this.lastOrderTime = Long.MAX_VALUE;
                } else {
                    this.lastSaleTime = Long.MAX_VALUE;
                }
                this.notifyAll();
            }
            this.completionLatch.countDown();
        }
    }

    private synchronized void process(Class<?> eventType, List<?> events) throws InterruptedException {
        if (eventType == Order.class) {
            this.processOrders(events);
        } else if (eventType == TimeAndSale.class) {
            this.processSales(events);
        } else {
            throw new IllegalArgumentException();
        }
    }

    private void processOrders(List<Order> orders) throws InterruptedException {
        for (Order order : orders) {
            if (this.useCompositeOrderOrQuote && order.getScope() != Scope.COMPOSITE || !this.useCompositeOrderOrQuote && order.getScope() != Scope.AGGREGATE) continue;
            long time = order.getTime();
            if ((time = Math.max(this.lastOrderTime, time)) < this.sessionStartTime) continue;
            ++this.orderCount;
            Slice slice = this.prepareSlice(order.getEventSymbol(), time);
            if (slice == null) continue;
            if (this.useCompositeOrderOrQuote) {
                slice.getBook().put(Long.valueOf(order.getOrderSide().getCode()), order);
            } else if (order.hasSize()) {
                slice.getBook().put(order.getIndex(), order);
            } else {
                slice.getBook().remove(order.getIndex());
            }
            slice.lastBookUpdate = time;
            this.lastOrderTime = time;
            this.waitOther(1);
            this.checkOld();
        }
    }

    private void processSales(List<TimeAndSale> sales) throws InterruptedException {
        for (TimeAndSale sale : sales) {
            long time;
            if (this.isRTOrTapeMode && !sale.isValidTick() || (time = sale.getTime()) < this.sessionStartTime) continue;
            if (time < this.lastSaleTime - this.oldThreshold) {
                System.out.println("!!! T&S too far back in time: " + sale);
                continue;
            }
            ++this.saleCount;
            Slice slice = this.prepareSlice(sale.getEventSymbol(), time);
            if (slice == null) continue;
            slice.sales.add(sale);
            slice.lastSaleUpdate = time = Math.max(time, this.lastSaleTime);
            this.lastSaleTime = time;
            this.waitOther(-1);
            this.checkOld();
        }
    }

    private Slice prepareSlice(String symbol, long time) {
        List<Slice> sl;
        if (this.startTime == 0L && time != 0L) {
            this.startTime = (time - this.oldThreshold) / this.period * this.period;
        }
        if (this.oldestActiveSliceTime == 0L && this.startTime != 0L) {
            this.oldestActiveSliceTime = this.startTime - this.oldThreshold;
        }
        if ((sl = this.allSlices.get(symbol)) == null) {
            if (this.oldestActiveSliceTime == 0L) {
                return null;
            }
            sl = new ArrayList<Slice>();
            this.allSlices.put(symbol, sl);
            if (this.zeroFillFromStartTimeToFirstTick) {
                sl.add(new Slice(symbol, this.startTime / this.period * this.period, this.period));
            } else {
                sl.add(new Slice(symbol, Math.max(this.oldestActiveSliceTime, time) / this.period * this.period, this.period));
            }
        }
        Slice lastSlice = sl.get(sl.size() - 1);
        while (lastSlice.endTime <= time) {
            lastSlice = new Slice(lastSlice);
            sl.add(lastSlice);
        }
        int i = sl.size();
        while (--i >= 0) {
            Slice s = sl.get(i);
            if (s.startTime > time || time >= s.endTime) continue;
            return s;
        }
        while (time < sl.get((int)0).startTime) {
            Slice slice;
            Slice next = sl.get(0);
            next.prev = slice = new Slice(symbol, next.startTime - this.period, this.period);
            sl.add(0, slice);
        }
        return sl.get(0);
    }

    private void waitOther(int direction) throws InterruptedException {
        if (this.waitThreshold == Long.MAX_VALUE) {
            return;
        }
        while ((this.lastOrderTime - this.lastSaleTime) * (long)direction > this.waitThreshold) {
            this.wait(10000L);
        }
        if ((this.lastSaleTime - this.lastOrderTime) * (long)direction > this.waitThreshold) {
            this.notifyAll();
        }
    }

    private void checkOld() {
        boolean hasMore;
        long time;
        long l = time = this.waitThreshold == Long.MAX_VALUE ? Math.max(this.lastOrderTime, this.lastSaleTime) : Math.min(this.lastOrderTime, this.lastSaleTime);
        if (time < this.oldestActiveSliceTime + this.oldThreshold) {
            return;
        }
        this.oldestActiveSliceTime = Math.min(this.endTime, (time - this.oldThreshold + this.period) / this.period * this.period);
        Object[] symbols = this.allSlices.keySet().toArray(new String[this.allSlices.size()]);
        Arrays.sort(symbols);
        do {
            hasMore = false;
            ArrayList<Slice> done = new ArrayList<Slice>(this.allSlices.size());
            for (Object symbol : symbols) {
                List<Slice> sl = this.allSlices.get(symbol);
                if (sl.get((int)0).startTime >= this.oldestActiveSliceTime) continue;
                Slice s = sl.remove(0);
                if (this.startTime <= s.startTime && s.startTime <= this.endTime) {
                    done.add(s);
                }
                if (sl.isEmpty()) {
                    sl.add(new Slice(s));
                }
                if (sl.get((int)0).startTime >= this.oldestActiveSliceTime) continue;
                hasMore = true;
            }
            this.slicesDone(done);
            this.doneSlices += (long)done.size();
        } while (hasMore);
    }

    protected void slicesDone(List<Slice> slices) {
        for (Slice slice : slices) {
            Session session;
            StringBuilder output = new StringBuilder(slice.symbol + ", " + TimeFormat.DEFAULT.format(slice.startTime) + ", ");
            if (this.schedule != null && !(session = this.schedule.getSessionByTime(slice.startTime)).isTrading()) {
                slice.isValid = false;
            }
            ArrayList<Order> sortedBook = new ArrayList<Order>(slice.getBook().values());
            Collections.sort(sortedBook, (o1, o2) -> Double.compare(o1.getPrice(), o2.getPrice()));
            StringBuilder bbo = new StringBuilder();
            StringBuilder bookAvgPrices = new StringBuilder();
            boolean oneSidedBook = true;
            for (int i = 0; i < sortedBook.size() - 1; ++i) {
                Order o12 = sortedBook.get(i);
                Order o22 = sortedBook.get(i + 1);
                if (o12.getPrice() >= o22.getPrice()) {
                    slice.isValid = false;
                }
                if (o12.getOrderSide() == Side.SELL && o22.getOrderSide() == Side.BUY) {
                    slice.isValid = false;
                    continue;
                }
                if (o12.getOrderSide() != Side.BUY || o22.getOrderSide() != Side.SELL) continue;
                bbo.append(o12.getPrice());
                bbo.append(", ");
                bbo.append(o12.getSizeAsDouble());
                bbo.append(", ");
                bbo.append(o22.getPrice());
                bbo.append(", ");
                bbo.append(o22.getSizeAsDouble());
                bbo.append(", ");
                oneSidedBook = false;
                if (!this.calcAvgBookPrices) continue;
                for (int j = 0; j < 5; ++j) {
                    bookAvgPrices.append(this.getAvgPriceForSize(this.avgBookSizes[j], Side.BUY, sortedBook, i));
                    bookAvgPrices.append(this.getAvgPriceForSize(this.avgBookSizes[j], Side.SELL, sortedBook, i));
                }
            }
            if (oneSidedBook && sortedBook.size() > 0) {
                Order o = sortedBook.get(0);
                if (o.getOrderSide() == Side.BUY) {
                    bbo.append(o.getPrice());
                    bbo.append(", ");
                    bbo.append(o.getSizeAsDouble());
                    bbo.append(", 0, 0, ");
                    if (this.calcAvgBookPrices) {
                        for (int j = 0; j < 5; ++j) {
                            bookAvgPrices.append("0, ");
                            bookAvgPrices.append(this.getAvgPriceForSize(this.avgBookSizes[j], o.getOrderSide(), sortedBook, 0));
                        }
                    }
                } else {
                    bbo.append("0, 0, ");
                    bbo.append(o.getPrice());
                    bbo.append(", ");
                    bbo.append(o.getSizeAsDouble());
                    bbo.append(", ");
                    if (this.calcAvgBookPrices) {
                        for (int j = 0; j < 5; ++j) {
                            bookAvgPrices.append(this.getAvgPriceForSize(this.avgBookSizes[j], o.getOrderSide(), sortedBook, 0));
                            bookAvgPrices.append("0, ");
                        }
                    }
                }
            }
            if (bbo.length() == 0) {
                bbo.append("0, 0, 0, 0, ");
            }
            if (this.calcAvgBookPrices && bookAvgPrices.length() == 0) {
                bookAvgPrices.append("0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ");
            }
            output.append((CharSequence)bbo);
            output.append((CharSequence)bookAvgPrices);
            double cumulativeSize = 0.0;
            double VWAP = 0.0;
            for (TimeAndSale sale : slice.sales) {
                cumulativeSize += sale.getSizeAsDouble();
                VWAP += sale.getPrice() * sale.getSizeAsDouble();
            }
            if (cumulativeSize > 0.0) {
                VWAP /= cumulativeSize;
            }
            output.append(VWAP);
            output.append(", ");
            output.append(cumulativeSize);
            output.append(", ");
            output.append(slice.isValid);
            this.outputWriteLn(output.toString());
        }
        this.outputFlush();
    }

    private String getAvgPriceForSize(long size, Side side, ArrayList<Order> sortedBook, int bboIndex) {
        int direction = 1;
        if (side == Side.BUY) {
            direction = -1;
        }
        if (side == Side.SELL) {
            ++bboIndex;
        }
        double cumulativeSize = 0.0;
        double avgPrice = 0.0;
        for (int i = bboIndex; i >= 0 && i < sortedBook.size() && cumulativeSize < (double)size; i += direction) {
            Order o = sortedBook.get(i);
            double orderSize = o.getSizeAsDouble();
            if (cumulativeSize + orderSize > (double)size) {
                orderSize = (double)size - cumulativeSize;
                cumulativeSize = size;
            } else {
                cumulativeSize += orderSize;
            }
            avgPrice += o.getPrice() * orderSize;
        }
        if (cumulativeSize > 0.0) {
            avgPrice /= cumulativeSize;
        }
        String result = "0, ";
        if (avgPrice > 0.0) {
            result = avgPrice + ", ";
        }
        return result;
    }

    public static class Slice {
        public final String symbol;
        public final long startTime;
        public final long endTime;
        public boolean isValid;
        public Slice prev;
        public Map<Long, Order> book;
        public final List<TimeAndSale> sales = new ArrayList<TimeAndSale>();
        public long lastBookUpdate;
        public long lastSaleUpdate;

        public Slice(String symbol, long startTime, long period) {
            this.symbol = symbol;
            this.startTime = startTime;
            this.endTime = startTime + period;
            this.isValid = true;
        }

        public Slice(Slice prev) {
            this(prev.symbol, prev.endTime, prev.endTime - prev.startTime);
            this.prev = prev;
        }

        public String toString() {
            return this.symbol + "{" + TimeFormat.DEFAULT.format(this.startTime) + " to " + TimeFormat.DEFAULT.format(this.endTime) + ", isValid = " + this.isValid + "}";
        }

        public Map<Long, Order> getBook() {
            if (this.book == null) {
                this.copyPrevBook();
            }
            return this.book;
        }

        private void copyPrevBook() {
            Slice head = this;
            while (head.prev != null) {
                head = head.prev;
            }
            Map<Long, Order> book = head.book;
            if (book == null) {
                head.book = book = new HashMap<Long, Order>();
            }
            head = this;
            while (head.prev != null) {
                head.book = new HashMap<Long, Order>(book);
                Slice prev = head.prev;
                head.prev = null;
                head = prev;
            }
        }
    }
}

