package com.dxfeed.acs;

import com.devexperts.logging.Logging;
import com.dxfeed.event.candle.Candle;
import com.dxfeed.event.candle.CandleSymbol;
import com.dxfeed.event.market.TimeAndSale;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Provides access to historical data of DXFeed Chart Server.
 */
public interface CandleService {
    static final Logging log = Logging.getLogging(CandleService.class);

    /**
     * Returns list of candles for the specified {@link CandleSymbol} and range of time.
     * The candles are ordered by {@link Candle#getTime() time} in the collection.
     * <p>
     * Candles in the list are marked with event flags (see "Event Flag" section in {@link com.dxfeed.event.IndexedEvent}).
     * To represent empty response will be returned the list with single empty candle
     * with {@link com.dxfeed.event.IndexedEvent#REMOVE_EVENT} flag in addition.
     *
     * @param candleSymbol the candle symbol to request (see {@link CandleSymbol})
     * @param fromTime     the time, inclusive, to request events from (see {@link Candle#getTime() Candle.getTime}).
     * @param toTime       the time, inclusive, to request events to (see {@link Candle#getTime() Candle.getTime}).
     *                     Use {@link Long#MAX_VALUE Long.MAX_VALUE} to retrieve events without an upper limit on time.
     * @return the result of the request
     */
    public List<Candle> getCandles(CandleSymbol candleSymbol, long fromTime, long toTime);


    /**
     * Returns list of candles for the specified collection of {@link CandleSymbol} and range of time.
     * The candles are ordered by {@link Candle#getTime() time} in the collection and if the time are equal - by
     * {@link CandleSymbol#toString()}.
     * <p>
     * Candles in the list are marked with event flags (see "Event Flag" section in {@link com.dxfeed.event.IndexedEvent})
     * for each particular symbol. If there are no candles for the symbol the result will be represented as single empty candle
     * with {@link com.dxfeed.event.IndexedEvent#REMOVE_EVENT} flag in addition.
     *
     * @param symbols  the list of candle symbols to request (see {@link CandleSymbol})
     * @param fromTime the time, inclusive, to request events from (see {@link Candle#getTime() Candle.getTime}).
     * @param toTime   the time, inclusive, to request events to (see {@link Candle#getTime() Candle.getTime}).
     *                 Use {@link Long#MAX_VALUE Long.MAX_VALUE} to retrieve events without an upper limit on time.
     * @return the result of the request
     */
    public List<Candle> getCandles(Collection<CandleSymbol> symbols, long fromTime, long toTime);

    /**
     * Returns list of candles for the specified collection of {@link CandleSymbol} and range of time.
     * The candles are ordered by {@link Candle#getTime() time} in the collection and if the time are equal - by
     * {@link CandleSymbol#toString()}.
     * <p>
     * Candles in the list are not marked with event flags (see "Event Flag" section in {@link com.dxfeed.event.IndexedEvent})
     * If there are no candles for some symbols the resulting list won't contain information about that.
     *
     * @param symbols  the list of candle symbols to request (see {@link CandleSymbol})
     * @param fromTime the time, inclusive, to request events from (see {@link Candle#getTime() Candle.getTime}).
     * @param toTime   the time, inclusive, to request events to (see {@link Candle#getTime() Candle.getTime}).
     *                 Use {@link Long#MAX_VALUE Long.MAX_VALUE} to retrieve events without an upper limit on time.
     * @return the result of the request
     */
    public List<Candle> getPlainCandles(Collection<CandleSymbol> symbols, long fromTime, long toTime);

    /**
     * Returns list of candles for the specified collection of {@link CandleSymbol} and range of time for specified
     * as of date. In other words - how candles were looked like at passed date in the past.
     * The candles are ordered by {@link Candle#getTime() time} in the collection and if the time are equal - by
     * {@link CandleSymbol#toString()}.
     * <p>
     * Candles in the list are marked with event flags (see "Event Flag" section in {@link com.dxfeed.event.IndexedEvent})
     * for each particular symbol. If there are no candles for the symbol the result will be represented as single empty candle
     * with {@link com.dxfeed.event.IndexedEvent#REMOVE_EVENT} flag in addition.
     *
     * @param symbols  the list of candle symbols to request (see {@link CandleSymbol})
     * @param fromTime the time, inclusive, to request events from (see {@link Candle#getTime() Candle.getTime}).
     * @param toTime   the time, inclusive, to request events to (see {@link Candle#getTime() Candle.getTime}).
     *                 Use {@link Long#MAX_VALUE Long.MAX_VALUE} to retrieve events without an upper limit on time.
     * @param asOfYmd  the integer value in yyyymmdd format to specify as of date
     * @return the result of the request
     */
    public List<Candle> getCandlesAsOf(Collection<CandleSymbol> symbols, long fromTime, long toTime, int asOfYmd);

    /**
     * Returns list of candles for the particular {@link CandleSymbol} and with specified count.
     * The candles are ordered by {@link Candle#getTime() time} in the collection. Number of candles in the result
     * collection may be less than {@code count} if there is no data to build more.
     * <p>
     * Candles in the list are marked with event flags (see "Event Flag" section in {@link com.dxfeed.event.IndexedEvent}).
     * To represent empty response will be returned the list with single empty candle
     * with {@link com.dxfeed.event.IndexedEvent#REMOVE_EVENT} flag in addition.
     *
     * @param candleSymbol the candle symbol to request (see {@link CandleSymbol})
     * @param count        number of candles to request
     * @return the result of the request
     * @deprecated will be removed in future releases
     */
    @Deprecated
    public default List<Candle> getLastCandles(CandleSymbol candleSymbol, int count) {
        log.error("getLastCandles call, will be removed in the future");
        return new ArrayList<>();
    }

    /**
     * Finds {@link TimeAndSale} events which have the the same time as passed one or closest to it.
     *
     * @param symbols {@link List} of {@link CandleSymbol} to request.
     * @param time    UTC time in milliseconds
     * @return {@link List} of {@link TimeAndSale} events with time equal to passed {@code time} or closest from the past
     * @deprecated subject for removal. Will return empty collection always.
     */
    @Deprecated
    public default List<TimeAndSale> getTimeAndSaleAtTime(List<CandleSymbol> symbols, long time) {
        log.error("getTimeAndSaleAtTime call, will be removed in the future");
        return new ArrayList<>();
    }

}
