package com.devexperts.mdd.news.event;

import com.devexperts.io.IOUtil;
import com.devexperts.util.TimeFormat;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Contains news details including time, title, news source, etc.
 */
public class NewsSummary implements Serializable, Comparable<NewsSummary>
{
    private static final long serialVersionUID = 0L;

    // Symbol used for stream subscription
    public static final String MESSAGE_SYMBOL = "NEWS";

    /**
     * Maximum length of title string.
     */
    public static final int MAX_TITLE_LENGTH = 32700;

    private NewsKey key;
    private String sourceId;
    private long time;
    private String title;
    private String source;

    private transient Map<String, Collection<String>> tags;

    public NewsSummary(NewsKey key, String sourceId, long time, String title, String source,
        Map<String, Collection<String>> tags)
    {
        this.key = key;
        this.sourceId = sourceId;
        this.time = time;

        if (title != null) {
            this.title = title.length() > MAX_TITLE_LENGTH ? title.substring(0, MAX_TITLE_LENGTH) : title;
        }
        this.source = source;
        this.tags = tags;
    }

    public NewsSummary(NewsSummary summary, Collection<String> filterTags) {
        this(summary.key, summary.sourceId, summary.time, summary.title, summary.source, cloneTags(summary, filterTags));
    }

    /**
     * Returns news key for identifying news.
     * @return news key.
     */
    public NewsKey getKey() {
        return key;
    }

    public void assignKey(NewsKey key) {
        if (!this.key.equals(NewsKey.FIRST_KEY))
            throw new IllegalStateException("key is already assigned");
        this.key = key;
    }

    /**
     * Returns external ID assigned by news provider.
     * @return external ID.
     */
    public String getSourceId() {
        return sourceId;
    }

    /**
     * Returns news time.
     * @return news time.
     */
    public long getTime() {
        return time;
    }

    /**
     * Returns news title.
     * @return news title.
     */
    public String getTitle() {
        return title;
    }

    /**
     * Returns news source provider.
     * @return news provider
     */
    public String getSource() {
        return source;
    }

    /**
     * Returns set of tags available for the news
     * @return set of tag names.
     */
    public Set<String> getTagNames() {
        return tags.keySet();
    }

    /**
     * Returns values for the specified tag, or null if not found.
     * @param tag tag name.
     * @return values for the specified tag, or null if not found.
     */
    public Collection<String> getTagValues(String tag) {
        return tags.get(tag);
    }

    /**
     * Returns set of instrument symbols for this news.
     * @return set of instrument symbols.
     */
    public Collection<String> getSymbols() {
        return getTagValues(NewsTags.SYMBOLS);
    }

    private static Map<String, Collection<String>> cloneTags(NewsSummary summary, Collection<String> filterTags) {
        if (summary.getTagNames().isEmpty())
            return Collections.emptyMap();

        Map<String, Collection<String>> result = new HashMap<>();
        for (String tag : filterTags) {
            Collection<String> tagValues = summary.getTagValues(tag);
            if (tagValues != null) {
                result.put(tag, tagValues);
            }
        }
        return result;
    }

    // Serializable interface methods

    private void writeObject(java.io.ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();

        IOUtil.writeCompactInt(out, tags.size());
        for (String tag : getTagNames()) {
            IOUtil.writeUTFString(out, tag);
            Collection<String> values = getTagValues(tag);
            IOUtil.writeCompactInt(out, values.size());
            for (String value : values) {
                IOUtil.writeUTFString(out, value);
            }
        }
    }

    private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();

        int size = IOUtil.readCompactInt(in);
        if (size == 0) {
            tags = Collections.emptyMap();
        } else {
            tags = new HashMap<String, Collection<String>>(size);
            for (int i = 0; i < size; i++) {
                String tag = IOUtil.readUTFString(in);
                int tagSize = IOUtil.readCompactInt(in);
                if (tagSize == 0) {
                    tags.put(tag, Collections.<String>emptySet());
                } else if (tagSize == 1) {
                    String value = IOUtil.readUTFString(in);
                    tags.put(tag, Collections.<String>singleton(value));
                } else {
                    Collection<String> values = new ArrayList<String>(tagSize);
                    for (int j = 0; j < tagSize; j++) {
                        String value = IOUtil.readUTFString(in);
                        values.add(value);
                    }
                    tags.put(tag, values);
                }
            }
        }
    }

    public int compareTo(NewsSummary o) {
        return getKey().getCode().compareTo(o.getKey().getCode());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        } else if (o instanceof NewsSummary) {
            NewsSummary that = (NewsSummary)o;
            return this.getKey().equals(that.getKey());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return getKey().hashCode();
    }

    public String toString() {
        return "NewsSummary{" + key.getCode() + "/" + getSourceId() +
            ", time=" + TimeFormat.DEFAULT.format(getTime()) +
            ", title=" + title +
            ", source=" + source +
            "}";
    }
}
