package com.devexperts.mdd.news.model;

import com.devexperts.logging.Logging;
import com.devexperts.mdd.news.event.NewsBody;
import com.devexperts.mdd.news.event.NewsFilter;
import com.devexperts.mdd.news.event.NewsKey;
import com.devexperts.mdd.news.event.NewsSummary;
import com.devexperts.rmi.RMIEndpoint;
import com.devexperts.rmi.RMIException;
import com.devexperts.rmi.RMIExceptionType;
import com.devexperts.rmi.RMIOperation;
import com.devexperts.rmi.RMIRequest;
import com.dxfeed.model.ObservableListModel;
import com.dxfeed.promise.Promise;
import com.dxfeed.promise.PromiseHandler;

import java.util.List;

/**
 * Model for convenient News management.
 *
 * <h3>Sample usage</h3>
 * The following code will print and update last 5 news from "Business Wire" source:
 * <pre>
 * NewsModel model = new NewsModel(endpoint);
 * model.getNewsList().addListener(new ObservableListModelListener&lt;NewsSummary&gt;() {
 *     public void modelChanged(Change&lt;? extends NewsSummary&gt; change) {
 *         for (NewsSummary news : change.getSource())
 *             System.out.println(news.getSourceId() + ": " + news.getTitle());
 *     }
 * });
 * model.setFilter(new NewsFilter("Business Wire", null, 5));
 * model.setLive(true);
 * </pre>
 *
 * <h3>Threads and locks</h3>
 *
 * <p>Listeners are invoked in the context of the corresponding {@link RMIEndpoint}
 * {@link com.devexperts.rmi.RMIEndpoint#getDefaultExecutor() executor} and the corresponding notification
 * is guaranteed to never be concurrent, even though it may happen from different
 * threads if executor is multi-threaded.
 */
//FIXME Will be changed to support new developments in dxFeed
//TODO Use DXEndpoint instead of RMIEndpoint
public class NewsModel implements Runnable
{
    private static final RMIOperation<String> GET_NEWS_CONTENTS = RMIOperation.valueOf(
        NewsService.class.getName(), String.class, "getNewsContents", NewsKey.class);

    private static final RMIOperation<NewsList> FIND_NEWS_FOR_FILTER = RMIOperation.valueOf(
        NewsService.class.getName(), NewsList.class, "findNewsForFilter", NewsFilter.class, NewsKey.class);

    private final RMIEndpoint endpoint;
    private RMIRequest<NewsList> request;

    private volatile boolean live;
    private volatile boolean running;
    private volatile NewsFilter filter = NewsFilter.getEmptyFilter();

    private volatile NewsKey lastKey = NewsKey.FIRST_KEY;
    private ObservableNewsList newsList = new ObservableNewsList();

    /**
     * Creates model with the specified RMI endpoint.
     * @param endpoint RMI endpoint to connect to
     */
    public NewsModel(RMIEndpoint endpoint) {
        this.endpoint = endpoint;
    }

    /**
     * Returns currently active news filter.
     * @return current news filter.
     */
    public NewsFilter getFilter() {
        return filter;
    }

    /**
     * Sets news filter.
     * @param filter news filter to use
     */
    public void setFilter(NewsFilter filter) {
        if (filter == null)
            throw new NullPointerException("filter is null");

        this.filter = filter;
        this.newsList.setLimit(filter.getLimit());
        if (request != null) {
            lastKey = NewsKey.FIRST_KEY;
            request.cancelOrAbort();

            newsList.beginChange();
            newsList.clear();
            newsList.endChange();
        }
    }

    /**
     * Returns whether news model is "live", i.e. receives news updates.
     * @return flag, indicating "live" status of the model.
     */
    public boolean isLive() {
        return live;
    }

    /**
     * Sets news model "live" status of the model. When model is "live"
     * it receives news updates from the news server.
     * @param live flag, indicating "live" status of the model.
     */
    public void setLive(boolean live) {
        if (this.live == live)
            return;
        this.live = live;

        if (shouldRun()) {
            endpoint.getClient().getDefaultExecutor().execute(this);
        } else if (request != null) {
            request.cancelOrAbort();
        }
    }

    /**
     * Returns an observable list of news which is updated on news updates.
     * @return ordered list of the latest news.
     */
    public ObservableListModel<NewsSummary> getNewsList() {
        return newsList;
    }

    /**
     * Returns news body for the specified news summary.
     * @param news news summary.
     * @return promise that will either return {@link NewsBody} or throw {@link NewsNotFoundException}.
     */
    public Promise<NewsBody> getNewsBody(final NewsSummary news) {
        RMIRequest<String> newsRequest = endpoint.getClient().createRequest(null, GET_NEWS_CONTENTS, news.getKey());
        Promise<String> promise = newsRequest.getPromise();
        newsRequest.send();
        return wrap(news, promise);
    }

    // Implementation

    final Logging log = Logging.getLogging(getClass());

    public void run() {
        while (shouldRun() && !Thread.currentThread().isInterrupted()) {
            running = true;
            try {
                request = endpoint.getClient().createRequest(null, FIND_NEWS_FOR_FILTER, filter, lastKey);
                request.send();
                final NewsList news = request.getBlocking();
                request = null;

                processNews(news);
            } catch (RMIException e) {
                if (e.getType() != RMIExceptionType.CANCELLED_BEFORE_EXECUTION &&
                    e.getType() != RMIExceptionType.CANCELLED_DURING_EXECUTION &&
                    e.getType() != RMIExceptionType.CANCELLED_AFTER_EXECUTION)
                    log.debug("RMI Exception: " + e.getType().getMessage());
            } catch (Throwable e) {
                log.debug("Exception while receiving news", e);
            } finally {
                running = false;
            }
        }
    }

    private void processNews(NewsList remoteNews) {
        lastKey = remoteNews.getLastKey();
        List<NewsSummary> list = remoteNews.getNews();

        newsList.beginChange();
        for (int i = list.size(); --i >= 0; ) {
            newsList.addChange(list.get(i));
        }
        newsList.endChange();
    }

    private boolean shouldRun() {
        return !running && live;
    }

    private static Promise<NewsBody> wrap(final NewsSummary summary, final Promise<String> promise) {
        final Promise<NewsBody> result = new Promise<NewsBody>();

        result.whenDone(new PromiseHandler<NewsBody>() {
            public void promiseDone(Promise<? extends NewsBody> p) {
                if (p.isCancelled())
                    promise.cancel();
            }
        });
        promise.whenDone(new PromiseHandler<String>() {
            public void promiseDone(Promise<? extends String> promise) {
                if (promise.hasResult())
                    result.complete(new NewsBody(summary, promise.getResult()));
                else
                    result.completeExceptionally(promise.getException());
            }
        });

        return result;
    }
}
