package com.devexperts.mdd.news.model;

import com.devexperts.mdd.news.event.NewsFilter;
import com.devexperts.mdd.news.event.NewsSummary;
import com.dxfeed.model.ObservableListModel;
import com.dxfeed.model.ObservableListModelListener;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Implementation of the observable list model for news.
 * It holds limited number of news sorted by time starting with the latest news.
 */
class ObservableNewsList extends AbstractList<NewsSummary>
    implements ObservableListModel<NewsSummary>
{
    private static final Comparator<NewsSummary> REVERSE_COMPARATOR = Collections.reverseOrder();

    private List<ObservableListModelListener<? super NewsSummary>> listeners =
        new CopyOnWriteArrayList<ObservableListModelListener<? super NewsSummary>>();

    private int limit;
    private ArrayList<NewsSummary> news = new ArrayList<NewsSummary>(NewsFilter.DEFAULT_LIMIT);
    private boolean changed;

    // AbstractList implementation

    @Override
    public NewsSummary get(int index) {
        return news.get(index);
    }

    @Override
    public int size() {
        return news.size();
    }

    @Override
    public void clear() {
        changed = true;
        news.clear();
    }

    // ObservableListModel interface implementation

    public void addListener(ObservableListModelListener<? super NewsSummary> listener) {
        listeners.add(listener);
    }

    public void removeListener(ObservableListModelListener<? super NewsSummary> listener) {
        listeners.remove(listener);
    }

    // Utility methods

    void setLimit(int limit) {
        if (limit <= 0)
            throw new IllegalArgumentException("limit must be positive: " + limit);
        this.limit = limit;

        if (news.size() > limit) {
            while (news.size() > limit)
                news.remove(news.size() - 1);
            fireModelChanged();
        }
    }

    protected void beginChange() {
        changed = false;
    }

    protected boolean addChange(NewsSummary newsSummary) {
        if (newsSummary == null)
            throw new NullPointerException("newsSummary");

        int index = Collections.binarySearch(news, newsSummary, REVERSE_COMPARATOR);
        if (index >= 0)
            return false;

        news.add(-index - 1, newsSummary);
        while (news.size() > limit)
            news.remove(news.size() - 1);

        changed = true;
        return true;
    }

    protected void endChange() {
        if (changed)
            fireModelChanged();
    }

    protected void fireModelChanged() {
        ObservableListModelListener.Change<? extends NewsSummary> change =
            new ObservableListModelListener.Change<NewsSummary>(this);
        for (ObservableListModelListener<? super NewsSummary> listener : listeners) {
            listener.modelChanged(change);
        }
    }
}
