package com.devexperts.mdd.news.event;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static java.util.Collections.emptySet;
import static java.util.Optional.ofNullable;

/**
 * Filter for news events.
 *
 * <p>Sample builder usage:
 * <pre><tt>
 *     NewsFilter filter = NewsFilter.newBuilder()
 *         .withSources("Business Wire")
 *         .withOrigins(NewsOrigin.valueOf("mt", "CBS"))
 *         .withSymbols("AAPL", "MSFT")
 *         .withLimit(50)
 *         .withTimeout(60_000L)
 *         .build();
 * </tt></pre>
 */
public class NewsFilter implements Serializable {

    private static final long serialVersionUID = 1L;

    /** @see NewsFilter#getFeeds() */
    @Deprecated
    public static final String FILTER_FEED = "feed";
    /** @see NewsFilter#getSources() */
    @Deprecated
    public static final String FILTER_SOURCE = "source";
    /** @see NewsFilter@getOrigins() */
    public static final String FILTER_ORIGIN = "origin";
    /** @see NewsFilter#getSymbols() */
    public static final String FILTER_SYMBOL = "symbol";
    /** @see NewsFilter#getLimit() */
    public static final String FILTER_LIMIT = "limit";
    /** @see NewsFilter#getDateLimit */
    public static final String FILTER_DATE_LIMIT = "dateLimit";
    /** @see NewsFilter#getIntervalLimit */
    public static final String FILTER_INTERVAL_LIMIT = "intervalLimit";
    /** @see NewsFilter#getTimeout() */
    public static final String FILTER_TIMEOUT = "timeout";

    /** Set of all available fields for the filter. */
    public static final Set<String> FILTER_PARAMETERS = new HashSet<>();
    static {
        FILTER_PARAMETERS.add(NewsFilter.FILTER_FEED);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_SOURCE);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_SYMBOL);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_ORIGIN);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_LIMIT);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_DATE_LIMIT);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_INTERVAL_LIMIT);
        FILTER_PARAMETERS.add(NewsFilter.FILTER_TIMEOUT);
    }

    /** Empty filter with all params set by default. */
    public static final NewsFilter EMPTY_FILTER = newBuilder().build();

    /** Default limit for the news count returned in one chunk. */
    public static final int DEFAULT_LIMIT = 100;

    /** Default time period to wait while polling for news. */
    public static final long DEFAULT_TIMEOUT = 60 * 60 * 1000L; // 1 hour

    /** Default date limit - deep past */
    public static final long DEFAULT_DATE_LIMIT = Long.MIN_VALUE;

    /** Default interval limit - unlimited */
    public static final long DEFAULT_INTERVAL_LIMIT = Long.MAX_VALUE;

    // All filter parameters are stored in the map
    private final Map<String, Set<String>> params;

    // Number fields extracted for convenience
    private transient int limit;
    private transient long timeout;

    // Combined origins extracted for convenience
    private transient Set<NewsOrigin> origins;

    // History limits: exact date and interval from moment of request
    private transient long dateLimit;
    private transient long intervalLimit;

    protected NewsFilter(Map<String, Set<String>> params, int originLimits) {
        this.params = immutableMap(params);
        initialize(originLimits);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static NewsFilter limitFilter(int limit) {
        return newBuilder().withLimit(limit).build();
    }

    public static NewsFilter emptyFilter() {
        return EMPTY_FILTER;
    }

    // Java Bean API
    
    public Builder builder() {
        return new Builder().withParams(getParams());
    }

    /**
     * Returns set of feeds to filter on, or {@code null} otherwise.
     * <p>
     * These feeds will be converted to {@link #getOrigins origins} in form of {@code &lt;feed&gt;:*}.
     * @deprecated  Use {@link #getOrigins} instead.
     */
    @Deprecated
    public Set<String> getFeeds() {
        return getParam(FILTER_FEED);
    }

    /**
     * Returns set of sources to filter on, or {@code null} otherwise.
     * <p>
     * These sources will be converted to {@link #getOrigins origins} in form of {@code *:&lt;source&gt;:*}.
     * @deprecated  Use {@link #getOrigins} instead.
     */
    @Deprecated
    public Set<String> getSources() {
        return getParam(FILTER_SOURCE);
    }

    /**
     * Returns set of all origins to filter on, or {@code null} otherwise.
     * <p>
     * Origins will include all added origins and all {@link #getFeeds() feeds} and {@link #getSources() sources}
     * converted into origins.
     */
    public Set<NewsOrigin> getOrigins() {
        return origins;
    }


    /** Returns set of instrument symbols to filter on, or {@code null} otherwise. */
    public Set<String> getSymbols() {
        return getParam(FILTER_SYMBOL);
    }

    /**
     * Returns maximum number of news to return as requested by client.
     * Note that server can return smaller number than requested due to entitlements.
     */
    public int getLimit() {
        return limit;
    }

    /**
     * Returns date of the oldest news to return, as UTC time in ms (same as {@link System#currentTimeMillis()}).
     * Note that server can restrict history due to entitlements.
     */
    public long getDateLimit() {
        return dateLimit;
    }

    /**
     * Returns age of the oldest news to return, in milliseconds.
     * Note that server can restrict history due to entitlements.
     */
    public long getIntervalLimit() {
        return intervalLimit;
    }

    /**
     * Sets the timeout in millis to wait while polling for news.
     * Note that server can use smaller timeout than requested due to entitlements.
     */
    public long getTimeout() {
        return timeout;
    }

    /**
     * Returns all filter parameters as a map.
     */
    public Map<String, Set<String>> getParams() {
        return params;
    }

    public Set<String> getParam(String name) {
        return params.get(name);
    }

    public String toString() {
        return "NewsFilter{" + getParams() + "}";
    }

    // Serialization API

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        // This is only for this class, so 10 is ok
        initialize(10);
    }

    // Utility methods

    /** Initialize and validate all required fields. */
    protected void initialize(int originsLimit) {
        //TODO Remove limits to entitlements
        limit = (int) validPositiveNumber(FILTER_LIMIT, getParam(FILTER_LIMIT), DEFAULT_LIMIT);
        timeout = validPositiveNumber(FILTER_TIMEOUT, getParam(FILTER_TIMEOUT), DEFAULT_TIMEOUT);
        dateLimit = validNumber(FILTER_DATE_LIMIT, getParam(FILTER_DATE_LIMIT), DEFAULT_DATE_LIMIT);
        intervalLimit = validNumber(FILTER_INTERVAL_LIMIT, getParam(FILTER_INTERVAL_LIMIT), DEFAULT_INTERVAL_LIMIT);
        validateSet(FILTER_SYMBOL, getParam(FILTER_SYMBOL), 10);
        validateSet(FILTER_FEED, getParam(FILTER_FEED), 10);
        validateSet(FILTER_SOURCE, getParam(FILTER_SOURCE), 10);
        validateSet(FILTER_ORIGIN, getParam(FILTER_ORIGIN), originsLimit);

        Set<NewsOrigin> origins =
            calculateOrigins(getParam(FILTER_ORIGIN), getParam(FILTER_FEED), getParam(FILTER_SOURCE));
        this.origins = origins.isEmpty() ? null: Collections.unmodifiableSet(origins);
    }

    protected static Set<NewsOrigin> calculateOrigins(Set<String> os, Set<String> fs, Set<String> ss) {
        Set<NewsOrigin> origins = new LinkedHashSet<>();
        if (os != null && !os.isEmpty())
            os.stream().map(NewsOrigin::valueOf).forEachOrdered(origins::add);

        // Feeds cross sources
        // Old code worked like this:
        //   NewsEvent conforms to filter, if
        //     1) its Feed is in Filter's Feeds set (or set is empty)
        //     AND
        //     2) its Source is in Filter's Sources set (or set is empty)
        // So, if Feeds is (a, b) and Sources is (y, z) these origins will match:
        // a:y, a:z, b:y, b:z
        // It is Cartesian Product of Feeds and Sources, with special case
        // when empty set is '*' (full outer join in SQL parlance).

        if (fs == null && ss != null)
            fs = Collections.singleton("*");
        if (fs != null && ss == null)
            ss = Collections.singleton("*");
        if (fs != null && ss != null) {
            for (String f : fs)
                for (String s : ss)
                    origins.add(NewsOrigin.valueOf(f, s));
        }
        return origins;
    }

    private static Map<String, Set<String>> immutableMap(Map<String, Set<String>> map) {
        if (map == null || map.isEmpty())
            return Collections.emptyMap();
        LinkedHashMap<String, Set<String>> copy = new LinkedHashMap<>(map);
        copy.replaceAll((k, values) -> Collections.unmodifiableSet(new LinkedHashSet<>(values)));
        return Collections.unmodifiableMap(copy);
    }

    private static long validPositiveNumber(String name, Collection<String> values, long defValue) {
        long value = validNumber(name, values, defValue);
        if (value < 1) {
            throw new IllegalArgumentException("Illegal '" + name + "': " + value + " must be positive");
        }
        return value;
    }

    private static long validNumber(String name, Collection<String> values, long defValue) {
        if (values == null || values.isEmpty()) {
            return defValue;
        }
        long value;
        try {
            value = Long.parseLong(values.iterator().next());
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("Illegal '" + name + "': value is not a number");
        }
        return value;
    }

    @SuppressWarnings("SameParameterValue")
    private static void validateSet(String name, Collection<String> values, long maxValue) {
        if (values != null && values.size() > maxValue)
            throw new IllegalArgumentException("Illegal '" + name + "': filter is too large");
    }

    // Builder class for NewsFilter

    public static class Builder {
        private final Map<String, Set<String>> params = new LinkedHashMap<>();

        protected Builder() {}

        // Public API for known parameters

        public Builder withFeeds(String... feeds) {
            return withParam(FILTER_FEED, Arrays.asList(feeds));
        }

        public Builder withFeeds(Collection<String> feeds) {
            return withParam(FILTER_FEED, feeds);
        }

        public Builder withSources(String... sources) {
            return withParam(FILTER_SOURCE, Arrays.asList(sources));
        }

        public Builder withSources(Collection<String> sources) {
            return withParam(FILTER_SOURCE, sources);
        }

        public Builder withOrigins(NewsOrigin... origins) {
            return withParam(FILTER_ORIGIN, Arrays.stream(origins).map(NewsOrigin::toString).collect(Collectors.toList()));
        }

        public Builder withOrigins(Collection<NewsOrigin> origins) {
            return withParam(FILTER_ORIGIN, ofNullable(origins).orElse(emptySet()).stream().map(NewsOrigin::toString).collect(Collectors.toList()));
        }

        public Builder withSymbols(String... symbols) {
            return withParam(FILTER_SYMBOL, Arrays.asList(symbols));
        }

        public Builder withSymbols(Collection<String> symbols) {
            return withParam(FILTER_SYMBOL, symbols);
        }

        public Builder withLimit(int limit) {
            return withParam(FILTER_LIMIT, String.valueOf(limit));
        }

        public Builder withTimeout(long timeout) {
            return withParam(FILTER_TIMEOUT, String.valueOf(timeout));
        }

        // Generic API methods

        public Builder withParams(Map<String, Set<String>> filters) {
            if (filters != null && !filters.isEmpty()) {
                filters.forEach(this::withParam);
            }
            return this;
        }

        public Builder withParam(String filter, Collection<String> values) {
            if (filter != null && !filter.isEmpty()) {
                if (values != null && !values.isEmpty()) {
                    params.put(filter, new LinkedHashSet<>(values));
                } else {
                    params.remove(filter);
                }
            }
            return this;
        }

        public Builder withParam(String filter, String value) {
            if (filter != null && !filter.isEmpty()) {
                if (value != null && !value.isEmpty()) {
                    params.put(filter, Collections.singleton(value));
                } else {
                    params.remove(filter);
                }
            }
            return this;
        }

        public NewsFilter build() {
            return new NewsFilter(params, 10);
        }
    }
}
