package com.devexperts.mdd.news.event;

import java.io.Serializable;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/**
 * This class describe pair of "news feed" and "news source", where "news source"
 * is subdivision of feed, as each feed provide {@link NewsEvent NewsEvents}
 * from one or more sources.
 * <p>
 * Origin can be exact, with fixed feed and source, and can be a mask, where
 * feed, source or both are set to '*'.
 * <p>
 * Each {@link NewsEvent NewsEvent's} origin is always exact.
 * <p>
 * Origins with masks can be used as {@link Predicate<NewsEvent> filters} for
 * {@link NewsEvent NewsEvents}.
 * <p>
 * String representation of origin is {@code &lt;feed&gt;:&lt;source&gt;}, where {@code &lt;feed&gt;}
 * is alphanumeric string which starts from letter and {@code &lt;source&gt;} is arbitrary string
 * of printable characters.
 * <p>
 * When {@link NewsOrigin} is created, empty {@code &lt;feed&gt;} or {@code &lt;source&gt;} is interpreted
 * as {@code *} and string without {@code :} (semicolon) is interpreted as {@code &lt;feed&gt;:*}.
 * <p>
 * Empty string is special value that means "nothing".
 */
public final class NewsOrigin implements Serializable {
    /**
     * {@link NewsOrigin} which matches any feed and source, {@code *:*}.
     */
    public final static NewsOrigin ANY = new NewsOrigin("*", "*");

    /**
     * {@link  NewsOrigin} which matches with nothing.
     */
    public final static NewsOrigin NOTHING = new NewsOrigin();

    private final String feed;
    private final String source;

    public static NewsOrigin valueOf(String origin) {
        Objects.requireNonNull(origin, "origin");
        if (origin.isEmpty())
            return NOTHING;
        int semicolon = origin.indexOf(':');
        // String without semicolon is feed only (like feed:*).
        if (semicolon == -1)
            semicolon = origin.length();
        String feed = origin.substring(0, semicolon);
        String source = semicolon + 1 < origin.length() ? origin.substring(semicolon + 1) : "*";
        return valueOf(feed, source);
    }

    public static NewsOrigin valueOf(String feed, String source) {
        checkFeedFormat(feed);
        Objects.requireNonNull(source, "source");
        if (feed.isEmpty() && source.isEmpty())
            return NOTHING;
        if (("*".equals(feed) || feed.isEmpty()) && ("*".equals(source) || source.isEmpty()))
            return ANY;
        return new NewsOrigin(feed.isEmpty() ? "*" : feed, source.isEmpty() ? "*" : source);
    }

    public static NewsOrigin valueOf(NewsEvent event) {
        Objects.requireNonNull(event, "event");
        return new NewsOrigin(event.getFeed(), event.getSource());
    }

    private NewsOrigin(String feed, String source) {
        this.feed = feed;
        this.source = source;
    }

    private NewsOrigin() {
        this.feed = "";
        this.source = "";
    }

    public String getFeed() {
        return feed;
    }

    public String getSource() {
        return source;
    }

    public boolean isFeedWildcard() {
        return "*".equals(feed);
    }

    public boolean isSourceWildcard() {
        return "*".equals(source);
    }

    public boolean isSourceUseWildCardPrefixFilter() {
        return source != null && source.endsWith("*") && !source.equals("*");
    }

    public boolean isWildcard() {
        return isFeedWildcard() && isSourceWildcard();
    }

    public boolean isExact() {
        return !isFeedWildcard() && !isSourceWildcard() && !feed.isEmpty() && !source.isEmpty();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        NewsOrigin that = (NewsOrigin) o;
        return feed.equals(that.feed) && source.equals(that.source);
    }

    @Override
    public int hashCode() {
        return Objects.hash(feed, source);
    }

    @Override
    public String toString() {
        return (feed.isEmpty() && source.isEmpty()) ? "" : feed + ":" + source;
    }

    private final static Pattern FEED_PATTERN = Pattern.compile("^[a-zA-Z][a-zA-Z0-9]*$");
    private static void checkFeedFormat(String feed) {
        Objects.requireNonNull(feed, "feed");
        if (feed.isEmpty() || "*".equals(feed))
            return;
        if (FEED_PATTERN.matcher(feed).matches())
            return;
        throw new IllegalArgumentException("Invalid feed name \"" + feed + "\"");
    }
}
