package com.dxfeed.orcs;

import com.devexperts.rmi.RMIEndpoint;
import com.devexperts.util.TimeFormat;
import com.devexperts.util.TimeUtil;
import com.dxfeed.event.candle.CandleSymbol;
import com.dxfeed.event.market.Order;
import com.dxfeed.event.market.OrderSource;
import com.dxfeed.event.market.Quote;
import com.dxfeed.orcs.api.AuthOrderSource;
import com.dxfeed.orcs.api.PriceLevelService;

import java.util.List;

/**
 * Sample class to demonstrate how to request orders and quotes via {@link com.dxfeed.orcs.api.PriceLevelService}.
 * It's possible to configure response validation via system property '-Dvalidate',
 * see {@link PriceLevelChecker#validate(List, long, boolean)}. This validation is performed additionally and doesnt'
 * belong to service interaction process.
 */
public class SamplePriceLevelServiceClient {
    private SamplePriceLevelServiceClient() {
    }

    public static void main(String[] args) {
        if (args.length < 5) {
            System.out.println("Usage: 'java -cp orcs-api\\lib\\* com.dxfeed.orcs.SamplePriceLevelServiceClient " +
                "<address> <symbol> <source> <from> <to>");
            System.out.println("	<address>  - server RMI address");
            System.out.println("	<symbol>   - symbol to retrieve, for example 'AAPL{=d,gr=s}'");
            System.out.println("	<source>   - source to retrieve, for example 'NTV'");
            System.out.println("	<from>     - datetime in yyyymmdd-hhmmss[+|-]zzzz format");
            System.out.println("	<to>       - datetime in yyyymmdd-hhmmss[+|-]zzzz format");
            System.exit(0);
        }
        String address = args[0];
        CandleSymbol candleSymbol = CandleSymbol.valueOf(args[1]);
        OrderSource source = OrderSource.valueOf(args[2]);
        // validation params
        long timeGapBound = TimeUtil.MINUTE;
        boolean printQuotes = true;

        RMIEndpoint endpoint = RMIEndpoint.createEndpoint(RMIEndpoint.Side.CLIENT);
        endpoint.connect(address);
        PriceLevelService service = endpoint.getClient().getProxy(PriceLevelService.class);
        // request authorized symbols
        AuthOrderSource authOrderSource = service.getAuthOrderSource();
        System.out.println("Authorized sources: " + authOrderSource.getByOrderSources().keySet());
        // request price levels
        long from = TimeFormat.GMT.parse(args[3]).getTime();
        long to = TimeFormat.GMT.parse(args[4]).getTime();
        long reqStart = System.currentTimeMillis();
        List<Order> orders = service.getOrders(candleSymbol, source.id(), from, to, "sample");
        System.out.println("Received " + orders.size() + " orders in " + (System.currentTimeMillis() - reqStart) + "ms");
        if (System.getProperty("validate") != null) {
            PriceLevelChecker.validate(orders, timeGapBound, printQuotes);
        }
        //request NBBO
        reqStart = System.currentTimeMillis();
        List<Quote> quotes = service.getQuotes(CandleSymbol.valueOf(candleSymbol.getBaseSymbol() + "{gr=s}"), from, to, "sample");
        System.out.println("Received " + quotes.size() + " quotes in " + (System.currentTimeMillis() - reqStart) + "ms");
        endpoint.close();
    }
}
